/************************************************************************/
/*	geometry.cc							*/
/*									*/
/*	This file holds functions describing the geometry of the	*/
/*	leaflets. In most cases, for a new geometry, all that should	*/
/*	need editing is this file (to put appropriate equations in) and	*/
/*	constant.h (which holds constants defined by the geometry)	*/
/************************************************************************/

#include <math.h>
#define	FIRST	1
#define	SECOND	2
#include "constant.h"

/************************************************************************/
/*	Returns maximum z value on curve defined by the intersection of	*/
/*	the leaflet and the confining cylinder. This is effectively the	*/
/*	stent height.							*/
/************************************************************************/
double	ZMax(
){
	return b*sqrt(1-((r-E)*(r-E))/(a*a));
}

/************************************************************************/
/*	This is an extra function added for this geometry for		*/
/*	convenience in a later function. It is not required for other	*/
/*	geometries.							*/
/************************************************************************/
double	Alpha(
double	z
){
	return E-a*sqrt(1-((z/b)*(z/b))) - H;
}

/************************************************************************/
/*	The maximum value of x (i.e. on the cylinder) at a given z	*/
/************************************************************************/
double	XMax(
double	z
){
	double	temp;
	double	alpha;

	alpha=Alpha(z);
	temp=(16*(alpha*alpha + (r*r)/3 - H*H))/3.0;
	return (2*H + sqrt(4*H*H+temp))*3.0/8.0;
}

/************************************************************************/
/*	x as a function of y and z.					*/
/************************************************************************/
double	xofyandz(
double	y,
double	z
){
	double	alpha;
	alpha=Alpha(z);
	return H + sqrt((y/k)*(y/k) + alpha*alpha);
}

/************************************************************************/
/*	The height (from z=0) of the curve defined by the intersection	*/
/*	of the leaflet and the cylinder, at some value of y.		*/
/************************************************************************/
double	IntersectHeight(
double	y
){
	double	x;
	double	tmp;

	x=sqrt(r*r-y*y);
	tmp=sqrt((x-H)*(x-H)-(r*r-x*x)/3);
	tmp=E-H-tmp;
	return b*sqrt(1-(tmp/a)*(tmp/a));
}	

/************************************************************************/
/*	z as a function of y and cyl_z. For many geometries this can	*/
/*	not be solved analytically, so a numerical approach is taken.	*/
/*	If the geometry to be used does have an analytical solution	*/
/*	here then use it instead.					*/
/*									*/
/*	This routine sets up an interval [0,zmax], and guesses z to	*/
/*	be at the midpoint of the interval. It then calculates		*/
/*	cyl_z(y,z) (which is usually an integral, which is why it is	*/
/*	difficult to invert) for this z and compares it to the cyl_z	*/
/*	passed to the routine. If cyl_z(y,z)>cyl_z(passed) then our	*/
/*	guess for z is too high and the z we want must be in the lower	*/
/*	half-interval. Similarly for other case and upper half-interval.*/
/*	The appropriate half-interval is chosen, and the process	*/
/*	repeated until the desired accuracy is achieved. This interval	*/
/*	halving technique works since cyl_z(y,z) is monatonic at const.	*/
/*	y.								*/
/************************************************************************/
double	zofyandcylz(
double	y,
double	cyl_z,
double	tolerance
){
	int	nsteps,i;
	double	z,cylztmp;
	double	zlow,zhigh;
	double	L,h,offset;
	extern double	Length(double(*)(double,double,int),int,double,double,double,int);
	extern double	dxofyandz(double, double, int);
	extern double	IntersectHeight(double);

	h=IntersectHeight(y);
	L=Length(dxofyandz,FIRST,y,0,h,50);
	offset=L-h;
	nsteps=(int) (log(zmax/tolerance)/log(2)) + 1;
	zlow=0;
	zhigh=zmax;
	for(i=0;i<nsteps;i++){
		z=(zhigh+zlow)/2;
		cylztmp=Length(dxofyandz,FIRST,y,0,z,50)-offset;
		if(cylztmp>cyl_z){
			zhigh=z;
		}
		if(cylztmp<cyl_z){
			zlow=z;
		}
		if(cylztmp==cyl_z) return z;
	}
	return z;
}


/************************************************************************/
/*	dxofyandz. This describes the derivative of x wrt y or z	*/
/*	(at constant z or y respectively). This 		 	*/
/*	function also takes an argument called const_arg, which is	*/
/*	FIRST if the first argument is to be held constant for the	*/
/*	derivative, and SECOND if the second argument is to be held	*/
/*	constant. For example						*/
/*		dxofyandz(y,z,FIRST)					*/
/*	should return							*/
/*		d       |						*/
/*		- x(y,z)|						*/
/*	       dz       |y						*/
/*	and								*/
/*		dxofyandz(y,z,SECOND)					*/
/*	returns								*/
/*		d       |						*/
/*		- x(y,z)|						*/
/*	       dy       |z						*/
/************************************************************************/
double	dxofyandz(
double	y,
double	z,
int	const_arg
){
	double	alpha,dalpha;

	alpha=-a*sqrt(1-((z/b)*(z/b)))+E-H;
	if(const_arg==FIRST){
		dalpha=(0-a)*z/(b*b*sqrt(1-(z/b)*(z/b)));
		return dalpha*alpha/sqrt((y/k)*(y/k)+alpha*alpha);
	}
	if(const_arg==SECOND){
		return y/(k*k*sqrt((y/k)*(y/k)+alpha*alpha));
	}
}
