/************************************************************************/
/*	former.cc							*/
/*	the main part of the program					*/
/************************************************************************/
#include <stream.h>
#include <math.h>
#define	PI	3.14159265358979323846
#define	FIRST	1
#define	SECOND	2
#define	CYL_NZSTEPS	20	// no. of steps to take along cylinder
				// z-axis.
#define	NYSTEPS		100	// no. of steps (in y direction) to use
				// when integrating length at constant
				// cylinder-z
#define	ZTOL		0.00001 // Tolerance to which z must be found
				// from cylz (included becuase for many
				// geometries z(y,cylz) cannot be found
				// symbolically, so need to use a numerical
				// technique)
#include "constant.h"		// constants used by the program to do with
				// the geometry

/************************************************************************/
/*      These are constants the whole program needs to know about       */
/*	(the following are values for geometry 1).			*/
/************************************************************************/
//a=22.032;       major radius of ellipse
//b=15.574;       minor radius of ellipse
//E=22.332;       ellipse offset
//H=0.23;         hyperbola offset
//r=11.2;         former radius
//zmax=13.44;     stent height
//k=1.7320508;    this is tan(60 degrees)

double	zmax;

int	main()
{
/************************************************************************/
/*	In the following the convention that all coordinate variables	*/
/*	in the coordinate system after deformation onto the surface of	*/
/*	the cylinder are prefixed by cyl_, variables in the leaflet	*/
/*	coordinate system carry no prefix.				*/
/************************************************************************/
	double	cyl_z;
	double 	cyl_zstep;
	int	cyl_nzsteps=CYL_NZSTEPS;
	double	path_length[CYL_NZSTEPS+1];	// array to hold const.
						// cylz path lengths
	double	radius[CYL_NZSTEPS+1];		//array to hold radii.
	double	cyl_zmax;
//	double	cyl_rad;
// 	double	cyl_circum;
	double	val_ymax;
	double	x,y,z;
	double	xold,yold,zold;
	double	xmax,ymax;
	double	ystep;
	int	nysteps=NYSTEPS;
	double	delta;
	int	i,j,k,kw;		//stepping variables for loops

	extern double	dxofyandz(double, double, int);
	extern double	zofyandcylz(double,double,double);
	extern double	xofyandz(double,double);
	extern double	Length(double(*)(double,double,int),int,double,double,double,int);
	extern double	XMax(double);
	extern double	ZMax();

	zmax=ZMax();
	cyl_zmax=zmax;
	cyl_z=cyl_zmax;
	cyl_zstep=(cyl_zmax-0)/cyl_nzsteps;
	radius[0]=r;
	path_length[0]=0.0;
	for(kw=1;kw<=cyl_nzsteps;kw++){
		cyl_z-=cyl_zstep;
		cerr << "Cyl z: "  << cyl_z << "\n";
		zold=cyl_z;
		xmax=XMax(zold); xold=xmax;
		ymax=sqrt(r*r - xmax*xmax); yold=ymax;
		ystep=ymax/nysteps;
		path_length[kw]=0.0;
		for(j=0;j<nysteps;j++){
			y=ymax-(ystep*(j+1));
			z=zofyandcylz(y,cyl_z,ZTOL);
			x=xofyandz(y,z);
			delta=sqrt((x-xold)*(x-xold)+(y-yold)*(y-yold)+(z-zold)*(z-zold));
			path_length[kw]+=delta;
			xold=x;
			yold=y;
			zold=z;
		}
		radius[kw]=(6*path_length[kw]+(2*PI-6*atan(ymax/xmax))*r)/(2*PI);
	}
	
	cout << "Warped z\tRadius\n";
	cout << cyl_zmax+2 << "\t" << r << "\n";
	for(i=0;i<=cyl_nzsteps;i++){
		cout << cyl_zmax - cyl_zstep*i << "\t" << radius[i] << "\n";
	}
	cout << -2 << "\t" << radius[cyl_nzsteps] << "\n";

	return 0;
}

void	rhs(
double	x,
double	*y,
double	*dydx,
double	(*dfdx)(double,double,int),
int	const_arg,
double	const_val
){
	double	temp;
	if(const_arg==FIRST){
		temp=(*dfdx)(const_val,x,const_arg);
		temp=temp*temp;
		dydx[0]=sqrt(1+temp);
	}
	else if(const_arg==SECOND){
		temp=(*dfdx)(x,const_val,const_arg);
		temp=temp*temp;
		dydx[0]=sqrt(1+temp);
	}
	return;
}

/****************************************************************************/
/* Runge-Kutte 4 routine taking:					    */
/*	int	num:	no. of equations.				    */
/*	double	*lhs:	pointer to lhs of equations.			    */
/*	double	*rhs:	pointer to rhs of equations.			    */
/*	double	x:	current value of integration variable.		    */
/*	double	step:	size of step to use.				    */
/*	void	*rhsfn:	pointer to fn. which evaluates rhs		    */
/* Plus modifications to the standard algorithm, for this program	    */
/*	double	*dfdx:	pointer to derivative of function to get length of  */
/*	int	const_arg:	which arg is constant in surface equation   */
/*	double	const_val:	value of constant arg			    */
/****************************************************************************/

void rkstp(
int	num,
double	*lhs,
double	*rhs,
double	x,
double	step,
void	(*rhsfn)(double,double*,double*,double(*)(double,double,int),int,double),
double	(*dfdx)(double,double,int),
int	const_arg,
double	const_val
){
	int q, w;
	double tlhs[10], tot[10];
	
	for(q = 0; q < num; q++){
		tlhs[q] = lhs[q];
		tot[q] = 0.0;
	}

	rhsfn(x, lhs, rhs, dfdx, const_arg, const_val);
	
	for(q = 0; q < num; q++){
		tot[q] += rhs[q];
		lhs[q] = tlhs[q] + (step / 2.0) * rhs[q];
	}
	
	rhsfn(x + (step / 2.0), lhs, rhs, dfdx, const_arg, const_val);
	
	for(q = 0; q < num; q++){
		tot[q] += 2.0 * rhs[q];
		lhs[q] = tlhs[q] + (step / 2.0) * rhs[q];
	}
	
	rhsfn(x + (step / 2.0), lhs, rhs, dfdx, const_arg, const_val);
	
	for(q = 0; q < num; q++){
		tot[q] += 2.0 * rhs[q];
		lhs[q] = tlhs[q] + step * rhs[q];
	}
	
	rhsfn(x + step , lhs, rhs, dfdx, const_arg, const_val);
	
	for(q = 0; q < num; q++){
		tot[q] += rhs[q];
	}

	for(q = 0; q < num; q++){
		lhs[q] = tlhs[q] + (step / 6.0)*tot[q];
	}

	return;
}

/************************************************************************/
/* Finds the length between two limits of a curve, given the derivative	*/
/* of the curve with respect to the integration variable		*/
/************************************************************************/

double	Length(
double	(*df)(double,double,int),
int	const_arg,
double	const_val,
double	lower_limit,
double	upper_limit,
int	num_steps
){
	int	i;
	double	step;
	double	l,dl,x;

	x=lower_limit;
	l=0.0;
	step=(upper_limit-lower_limit)/num_steps;
	for(i=0;i<num_steps;i++){
		rkstp(1,&l,&dl,x,step,rhs,df,const_arg,const_val);
		x+=step;
	}
	return l;
}

