#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "matops.h"

// This file contains matrix support operations
// and should be included in every program that
// does things with matrices

/*************************************
*  Error handler for matrixfunctions *
*************************************/
void nrerror (char error_text[])
{
   fprintf(stderr, "Numerical recipes run-time error...\n");
   fprintf(stderr, "%s\n",error_text);
   fprintf(stderr, "... Now exiting to system...\n");
   exit(1);
}

/*****************************************************
* Following group of functions serve to allocate     *
* and de-allocate arrays and vectors for matrix      *
* functions. Note that indices all start at 1!       *
*****************************************************/

int * ivector(int nl, int nh)
/*  Allocates a vector of ints with range [nl..nh] */
{
   int * v;
   v = (int *) malloc((nh-nl+1) * sizeof(int));
   if (!v) nrerror("Allocation failure in ivector()");
   return v-nl;
}

void free_ivector(int *v, int nl, int nh)
/* Frees the vector allocated by vector() */
{
   free((char *)(v+nl+nh-nh)); /* To disable spurious warning!*/
}

double * vector(int nl, int nh)
/*  Allocates a vector of doubles with range [nl..nh] */
{
   double * v;
   v = (double *) malloc((nh-nl+1) * sizeof(double));
   if (!v) nrerror("Allocation failure in vector()");
   return v-nl;
}

void free_vector(double *v, int nl, int nh)
/* Frees the vector allocated by vector() */
{
   free((char *)(v+nl+nh-nh)); /* To disable spurious warning!*/
}

double ** matrix(int nrl,int nrh,int ncl,int nch)
/* Allocates a matrix of doubles with range [nrl..nrh] [ncl..nch] */
{
  int i; double ** m;
                 /* Allocate pointers to rowscc32 */
  m=(double**) malloc((nrh-nrl+1)*sizeof(double*));
  if (!m) nrerror("Allocation failure 1 in matrix()");
  m -= nrl;
                 /* Allocate rows and set pointers to them */
  for(i=nrl;i<=nrh; i++)
  {
      m[i]=(double*)malloc((nch-ncl+1) * sizeof(double));
      if (!m[i]) nrerror("Allocation failure 2 in matrix()");
      m[i] -= ncl;
  }
  return m;
}

void free_matrix(double**m, int nrl,int nrh,int ncl, int nch)
/*  Frees the matrix allocated by matrix() */
{
   int i;
    for (i=nrh; i>=nrl; i--)  free ((char *) (m[i]+ncl+nch-nch)); /* To disable spurious warning!*/
    free ((char *) m+nrl);
}

/**********  End  of allocation/deallocation functions   ***********/


void matMult(double**a,double**b,double**c,int n)
{   /* Multiplies two square matrices a and b, and gives the result in c */
   int i,j,k;
   for(i=1; i<=n; i++)         /*row number*/
		for(j=1; j<=n; j++)    /*column number */
        {
			c[i][j] = 0.0;
			for (k=1; k<=n; k++)
			   c[i][j] += a[k][j]*b[i][k];
		}
}

void matAdd(double**a,double**b,double**c,int n)
{   /* Adds two square matrices a and b, and gives the result in c */
   int i,j;
    for(i=1; i<=n; i++)         /*row number*/
 		for(j=1; j<=n; j++)    /*column number */
           c[i][j] = (a[i][j]+b[i][j]);
}
void matSub(double**a,double**b,double**c,int n)
{   /* Subtracts two square matrices a and b, and gives the result in c */
   int i,j;
    for(i=1; i<=n; i++)         /*row number*/
 		for(j=1; j<=n; j++)    /*column number */
           c[i][j] = a[i][j]-b[i][j];
}

void matScale( double**a, double **b, double scaleFactor, int n)
{    /* Multiples every element in a by the scaleFactor
        and delivers the result in b
     */
   int i,j;
    for(i=1; i<=n; i++)         /*row number*/
 		for(j=1; j<=n; j++)    /*column number */
           b[i][j] = scaleFactor*a[i][j];
}


void matWeightedDifference (double **a, double **b,double **c,double weight,int n)
{    /* calculates weight * a - (1-weight) *b
        anddelivers the result in c
     */
	  int i,j;
	  double z = 1.0-weight;
	  for(i=1; i<=n; i++)         /*row number*/
 		for(j=1; j<=n; j++)    /*column number */
           c[i][j] = weight*a[i][j] - z*b[i][j];
}

void matDisplay(FILE * f,char * label, double ** a, int n)
{
	// Sends a matrix to the file
	int i,j;
	fprintf(f,"**************************************************\n    %s\n\n",label);
	for (i = 1; i<=n; i++)
	{
		for (j=1; j<=n; j++)  fprintf(f,"%6.3lf ",a[i][j]);
		fprintf(f,"\n");
    }
    fprintf(f,"**************************************************\n\n");
}

void display(char * label, double ** a, int n)
{
	//Displays a matrix on the screen
	int i,j;
	printf("%s\n",label);
	for (i = 1; i<=n; i++)
	{
		for (j=1; j<=n; j++)  printf("%7.4lf ",a[i][j]);
		printf("\n");
    }
    printf("\n\n");
}

void displayRectangle(char * label, double ** a, int rows,int cols)
{
	//Displays a non-square matrix on the screen
	int i,j;
		printf("\n%s\n\n",label);

	for (i = 1; i<=rows; i++)
	{
		for (j=1; j<=cols; j++)  printf("%7.4lf ",a[i][j]); printf("\n");
    }
    printf("\n\n");
}

void matDisplayRectangle(FILE * f,char * label, double ** a, int rows,int cols)
{
	//Displays a non-square matrix on the screen
	int i,j;
		fprintf(f,"    %s\n",label);
	for (i = 1; i<=rows; i++)
	{
		for (j=1; j<=cols; j++)fprintf(f,"%6.3lf ",a[i][j]);
		fprintf(f,"\n");
    }
    fprintf(f,"\n");
}
void randomMatrix(double **a, int n)
{
	 /* Puts a random n x n matrix into a */
   int i,j;
    for(i=1; i<=n; i++)         /*row number*/
 		for(j=i; j<=n; j++)    /*column number */
 		 a[i][j] =a[j][i]=  rand()/50000.0;
}

void matTranspose(double **a, double **b, int n)
{
	  //Transposes matrix a into b
	  int i,j;
	  for(i=1;i<=n;i++)
	  for(j=1;j<=n; j++)
	     b[i][j] = a[j][i];
}

int fact (int n)
{
   // finds n!
   if (n==0)
      return 1;
   else return n * fact(n-1);
}


double dotProduct(double * p,double* q, int n)
{
    // dotProduct works out the dot product of two vectors
    int a;
    double prod = 0.0;
    for(a=1; a<=n; a++) prod += p[a]*q[a];
    return prod;
}



void subVec (double * p, double *q, double j,int n )
{
   // subVec subtracts j times vector q from vector p
   int a;
   for (a=1; a<=n;a++) p[a] -= j*q[a];
}




void normalise (double * p, int n)
{
   // normalises vector p
   int a;
   double s = 0.0;
   for(a=1; a<=n; a++) s += p[a]*p[a];
   s=sqrt(s);
   for(a=1;a<=n; a++) p[a]/=s;
}

double trace(double ** x, int size)
{
	double v = 0.0;
	int a;
	for (a=1;a<=size; a++) v += x[a][a];
	return v;
}


void copyMatrix( double ** x, double ** y, int size)
{
	//copies a SQUARE matrix from x to y. Both must exist
	int a,b;
	for(a=1; a<=size;a++)
	  for(b=1; b<=size; b++)
	     y[a][b] = x[a][b];
}


/**********************************************
*   Tensor product of two matrices            *
*   The left-hand matrix is supplied in x.    *
*   It has rx rows and cx columns, Likewise   *
*   the right-hand matrix is supplied in y.   *
*   It has ry rows and cy columns. The result *
*   appears in prod, which is undefined on    *
*   entry.                                    *
*   Note that the final parameter is the      *
*   ADDRESS if the relevant matrix            *
*   'cos otherwise it duzznt work!            *
**********************************************/

void tensorProduct(double **x, int rx,int cx, double **y, int ry,int cy, double*** prod)
{
   int a,b,c,d;
 //  *prod = matrix(1, rx*ry, 1, cx*cy);
 //  printf("Made matrix\n");
   for (a= 1; a<= cx; a++)
      for (b = 1; b<=rx; b++)
      {
          int sc = cy*(a-1);
          int sr = ry*(b-1);
          for(c=1; c<= cy; c++)
             for(d=1;d<= ry; d++)
				 (*prod)[sc+c][sr+d] = x[a][b]*y[c][d];
      }
}



/*******************************************************
* Linear equation solution by Gauss-Jordan elimination *
* a[1..n][1..n] is an input matrix of n by n elements. *
* b[1..n][1..m] is an input matrix of size n by m      *
* containing the m right-hand vectors. On output, a is *
* replaced by its matrix inverse, and b is replaced by *
* the corresponding set of solution vectors.           *
* Copied from p.36 of Numerical Recipes in C           *
*******************************************************/

#define SWAP(a,b) {double temp = (a); (a) = (b); (b) = temp;}
void gaussj(double ** a, int n, double **b, int m)
{     //1
   int *indxc, *indxr, *ipiv;
   /* The integer arrays ipiv[1..n], indxc [1..n] and indxr[1..n]
      are used for bookkeeping on the pivoting
   */
   int i,icol,irow,j,k,l,ll,*ivector();
   double big,dum,pivinv;
   indxc = ivector(1,n);
   indxr = ivector(1,n);
   ipiv = ivector(1,n);
   for(j=1;j<=n;j++) ipiv[j] = 0;
   for(i=1; i<=n;i++)
   {   //2
       big = 0.0;
       for(j=1; j<=n; j++)
          if(ipiv[j] != 1)
             for(k=1; k<=n; k++)
             {  //3
                 if( ipiv[k]==0)
                 {  //4
                     if(fabs(a[j][k]) >= big)
                     {   //5
                        big=fabs(a[j][k]);
                        irow = j; icol = k;
                     }  //5
                  }   //4
                  else if (ipiv[k]>1) nrerror("GAUSSJ: Singular matrix-1");
              }   //3
        ++(ipiv[icol]);
        /*   We now have the pivot element, so we interchange rows,
             if needed, to put the pivot element on the diagonal.
             The columns are not physically interchanged, only
             relabeled: indxc[i] is the row in which that pivot
             element was originally located. If indxc[i] !=
             indxr[i] there is an implied column interchange.
             With this form of bookkeeping, the solution b's will
             end up in the correct order, and the inverse matrix
             will be scrambled by columns.
        */
        if (irow != icol)
        {  //3
           for(l=1; l<=n; l++) SWAP(a[irow][l],a[icol][l])
           for(l=1; l<=m; l++) SWAP(b[irow][l],b[icol][l])
        }   //3
        /* We are now ready to divide the pivot row by the pivot element,
           located at irow and icol
        */
        indxr[i] = irow;
        indxc[i] = icol;
        if(a[icol][icol] == 0.0) nrerror("GAUSSJ: Singular matrix-2");
        pivinv=1.0/a[icol][icol];
        a[icol][icol] = 1.0;
        for(l=1;l<=n; l++) a[icol][l] *= pivinv;
        for(l=1;l<=m; l++) b[icol][l] *= pivinv;
        /*  Next we reduce the rows .. except for the pivot one, of course */
        for(ll=1;ll<=n;ll++)
           if(ll != icol)
           {  //3
              dum = a[ll][icol];
              for(l=1;l<=n; l++) a[ll][l] -= a[icol][l]*dum;
              for(l=1;l<=m; l++) b[ll][l] -= b[icol][l]*dum;
           }  //3
       }  //2
       /* This is the end of the main loop over columns of the reduction.
          It only remains to unscramble the solution in view of the
          column interchanges. We do this by interchanging pairs of
          columns in the reverse order that the permutation was built up.
        */
        for(l=n; l>=1; l--)
        { //2
          if(indxr[l] != indxc[l])
              for(k=1; k<=n; k++)     /* This line added to code from the book */
                 SWAP(a[k][indxr[l]],a[k][indxc[l]]);
        }  //2
          free_ivector(ipiv,1,n);
          free_ivector(indxr,1,n);
          free_ivector(indxc,1,n);
}  //1


/****************************************************
*                                                   *
* This version of the Gram-Schmidt procedure takes  *
* ev vectors each of size n, which together span a  *
* space of ev dimensions. The function finds an     *
* orthonormal basis for the space. Each element of  *
* the basis is delivered as a linear combination of *
* the input vectors.                                *
*                                                   *
* The parameters are as follows:                    *
*                                                   *
*  matrix w(1,n,1,ev) holds the input vectors       *
*  matrix v(1,n,1,ev) delivers the output          *
*  n and ev are the dimensions,                     *
*  The function removes null vectors and returns    *
*  the effective number of dimensions               *
****************************************************/





void copyVector(double ** p,int j, double ** q, int k, int n)
{
	int x;
	for(x = 1; x<= n; x++)
	p[x][j] = q[x][k];
}

double myDotProduct(double ** p,int j, double ** q, int k, int n)
{
	int x;
	double z = 0;
	for(x=1; x<=n; x++)
	   z += p[x][j] * q[x][k];
	return z;
}

void takeAway(double ** p,int j, double ** q, int k, int n,double f)
{
	int x;
	for (x=1; x<=n; x++)
	    p[x][j] -= f*q[x][k];
}


/***************************************************
*  This version of the gram-schmidt procedure only *
*  works correctly if the vectors supplied are     *
*  linearly independent. We have to make sure that *
*  it is acually so,                               *
***************************************************/

int gramSchmidt2(double **w, double ** v, int n, int ev)
{   //1
    int a,b,c,d;double mult;
    for(a=1; a<= ev; a++)
    {
		copyVector(v,a,w,a,n);
		for(b=1; b<a; b++)
		{
			double g,h;

			g= myDotProduct(v,b,w,a,n);
			h = myDotProduct(v,b,v,b,n);
            if(h > 0)
            {
			   mult = g  /h ;
			   takeAway(v,a,v,b, n,mult);
		    }
		}
	}
	b=0;
	for(a=1; a<=ev;a++)
	{ //2
		int c;
		double j= myDotProduct (v,a,v,a,n);
		j = sqrt(j);
		if (fabs(j) > 0.0001)
		{
			b++;
		    for(c=1; c<=n; c++)
			 v[c][b] = v[c][a]/j;
		 }
	}  //2
    return b;
}  //1








