 /*************************************************
* This is module gcdm.c                          *
* The functions in the module build two density  *
* matrices, rho1 and rho2, for the configuration *
* where both program qbits are on the same GIVEN *
* great circle. Unlke other configurations that  *
* we consider, these matrices do not have a band *
* structure, and that limits the size of problem *
* that we can handle - up to 12 qbits in all.    *
*************************************************/

extern double ** rho1;
extern double ** rho2;


double integral2(int m, int n);
void makeDensityMatricesForGreatCircle(int p1,int d, int p2);



/*******************************************************
* Calculates the integral from 0 to 2pi of the         *
* function  cos^n(x)sin^m(x),using a recursive method. *
* This is used to set up the density matrix of the     *
* great circle case,                                   *
*******************************************************/

double integral2(int m, int n)

{
	if ((m+n)&1) printf("Bad parameters for integral2\n");
	if (m&1)       // Parameters are odd
	    return 0;
    if (n>0)      // General recursive rule
      return(integral2(m,n-2) - integral2(m+2,n-2));
    {             // Terminal case: integral(cos^n)(ax) = ((n-1)/n)*((n-3)/(n-2))... (1)/2)
		int j; double p=1.0;
		for(j=m;j>0;j-=2)
		  p = p*(j-1.0)/j;
		return p;

	}
}



/********************************************************
*                                                       *
* Function makeDensityMatricesForGreatCircle  produces  *
* two density matrices, rho1 and rho2, for the triad    *
* problem, but where there are p1 copies of the first   *
* program qbit, d copies of the data qbit, and p2       *
* of the second program qbit.                           *
*                                                       *
* To produce rho1, we assume that the data qbit is the  *
* same as the first program data qbit, and work out     *
* density matrices: xx for |phi1> ^ (p1+q) (where ^     *
* means tensor multiplication) and yy for |phi2>^(p2)   *
* Then we form the tensor product rho1 = xx^yy          *
* rho2 is formed the other way round. Note that we can  *
* make substantial savings if p1=p2 (as it often will   *
* be).                                                  *
* To work out the actual elements in each of the        *
* density matrices, consider element [a][b] where a and *
* b start at 1 (not zero!)                              *
* Calculate the number of 1's in  the binary represt-   *
* ation of (a-1) (say a1) and the number of 0's (a0)    *
* Do the sane for (b-1) giving b1 and b0. Then the      *
* element at [a][b] is the integral from 0 to 2pi       *
* of cos^(a0+b0)(x/2)sin^(a1+b1)(x/2) dx.               *
*                                                       *
* The powers add up to a0+a1+b0+b1,When the sum is odd, *
* the integral is zero. When the sum is even we can use *
a recursive formula as given above.                     *
********************************************************/


void makeDensityMatricesForGreatCircle(int p1,int d, int p2)
{
	int a,b, d1,d2,a0,a1,b0,b1;
	double **xx;
	double ** yy;
	d1 = power(2,(p1+d));
	d2 = power(2,p2);
	xx = matrix(1,d1,1,d1);
	yy = matrix(1,d2,1,d2);
	for(a=1;a<=d1;a++)
	  for(b=1;b<= d1; b++)
	  {
		  a1 = countOnes(a-1);
		  a0 = p1+d-a1;
		  b1 = countOnes(b-1);
		  b0=  p1+d-b1;
		  if ((a1+b1)&1)
		     xx[a][b] = 0;
		  else
			   xx[a][b] = integral2(a0+b0,a1+b1);
       }
     for(a=1;a<=d2;a++)
	   for(b=1;b<= d2; b++)
	   {
		  a1 = countOnes(a-1);
		  a0 = p2-a1;
		  b1 = countOnes(b-1);
		  b0=  p2-b1;
	      if ((a1+b1)&1)
			 yy[a][b] = 0;
		  else
			 yy[a][b] = integral2(a0+b0,a1+b1);
	   }
	   tensorProduct(xx,d1,d1,yy,d2,d2,&rho1);
	   if (p1==p2)
		   tensorProduct(yy,d2,d2,xx,d1,d1,&rho2);

		  else
		  {
			d2 = power(2,(p2+d));
			d1 = power(2,p1);
			xx = matrix(1,d1,1,d1);
			yy = matrix(1,d2,1,d2);
			for(a=1;a<=d1;a++)
			  for(b=1;b<= d1; b++)
			  {
				  a1 = countOnes(a-1);
				  a0 = p1-a1;
				  b1 = countOnes(b-1);
				  b0=  p1-b1;
				  if ((a1+b1)&1)
					 xx[a][b] = 0;
				  else
					 xx[a][b] = integral2(a0+b0,a1+b1);
				 }
			   for(a=1;a<=d2;a++)
			   for(b=1;b<= d2; b++)
			   {
				  a1 = countOnes(a-1);
				  a0 = p2+d-a1;
				  b1 = countOnes(b-1);
				  b0=  p2+d-b1;
				  if ((a1+b1)&1)
					 yy[a][b] = 0;
				  else
					 yy[a][b] = integral2(a0+b0,a1+b1);
				  }
			  tensorProduct(xx,d1,d1,yy,d2,d2,&rho2);
		  }
   }
