/*****************************************************************************
*  Function jacobi computes all the eigenvalues and eigenvectors of a REAL   *
*  SYMMETRIC matrix a[1..n][1..n] On output the elements of a above the      *
*  diagonal are destroyed.  d[1..n] returns the eigenvalues of a.            *
*  v[1..n][1..n] is a matrix whose columns contain, on output, the normalised*
*  eigenvectors of a. nrot returns the number of jacobi rotations that were  *
*  required.   Some utility functions are given above.                       *
*  In this code all array and matrix indices start at 1.                     *
*  Code is taken from Numerical Recipes pp 364-366                           *
*****************************************************************************/
#include "matops.h"

#define ROTATE(a,i,j,k,l) g=a[i][j];h=a[k][l];a[i][j]=g-s*(h+g*tau);\
     a[k][l]= h+s*(g-h*tau);

 void jacobi(double **a, int n, double d[], double **v,int * nrot)
 {   /* 1*/
     int j=0,iq=0,ip=0,i=0;
     double tresh=0.0, theta=0.0,tau=0.0,t=0.0,sm=0.0,s=0.0,h=0.0,g=0.0,c=0.0,*b,*z;
	 b=vector(1,n); z=vector(1,n);
     for(ip=1;ip<=n; ip++)     /* Initialise the identity matrix */
     {
        for (iq=1; iq <= n; iq++) v[ip][iq] = 0.0;
        v[ip][ip] = 1.0;
     }
     for(ip=1;ip<=n; ip++)    /*Initialise b and d to the diagonal of a */
     {
	    b[ip]=d[ip] = a[ip][ip];
        z[ip] = 0;
     }
     *nrot = 0;
     for (i=1; i<= 50; i++)    /* Iteration counter */
     {  /* 2 */
		 sm = 0.0;		/* Sum elements abouve the diagonal */
		 for (ip = 1; ip <= n-1; ip++)
		    for (iq = ip+1; iq<=n; iq++)
		      sm += fabs(a[ip][iq]);
  	      if (fabs(sm) < 1.0e-12)      /* Return */
		 {
			 free_vector (z,1,n);
			 free_vector(b,1,n);
			 return;
		 }
		 if (i<4)
		    tresh = 0.2*sm/(n*n);    /* On first three sweeps */
		 else
		     tresh = 0.0;
		 for (ip=1; ip <= n-1; ip++)
		 {   /* 3 */
			 for(iq=ip+1; iq <=n; iq++)
			 { /* 4 */
				 g=100.0*fabs(a[ip][iq]);
				 if (i>4  &&
				     (fabs(d[ip]+g) == fabs(d[ip])) &&
			         (fabs(d[iq]+g) == fabs(d[iq])))
			         a[ip][iq] = 0.0;
			     else
				 if (fabs(a[ip][iq]) > tresh)
			     {  /* 5 */
			        h = d[iq]-d[ip];
			        if ((fabs(h) +g) == fabs(h))
			           t= (a[ip][iq])/h;
			        else
			        {   /* 6 */
						  theta = 0.5*h/(a[ip][iq]);
						  t = 1.0/(fabs(theta)+sqrt(1.0+theta*theta));
						  if (theta < 0.0) t = -t;
				    }  /* 6 */
					c=1.0/sqrt(1+t*t);
				    s=t*c;
				    tau = s/(1.0+c);
				    h=t*a[ip][iq];
				    z[ip]-=h;
				    z[iq]+=h;
				    d[ip]-=h;
				    d[iq]+=h;
				    a[ip][iq] = 0.0;
				    for (j=1;j<=ip-1; j++)
				    {  ROTATE(a,j,ip,j,iq); }
				    for (j=ip+1;j<=iq-1; j++)
				    {  ROTATE(a,ip,j,j,iq); }
				    for (j=iq+1;j<=n; j++)
				    {  ROTATE(a,ip,j,iq,j); }
				    for (j=1;j<=n; j++)
				    {  ROTATE(v,j,ip,j,iq); }
				    ++(*nrot);
			   }   /* 5 */
	      } /* 4 */
       } /* 3 */
       for(ip = 1; ip <= n; ip++)
       {
		   b[ip]+= z[ip];
		   d[ip] = b[ip];
		   z[ip] = 0.0;
	   }
    } /* 2 */
    nrerror("Too many iterations in routine jacobi");
} /* 1 */


