/*
 * $Id: TextNode.java,v 1.4 1999/04/20 00:11:14 mode Exp $
 * 
 * Copyright (c) 1998-1999 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 */

package com.sun.xml.tree;


import java.io.Writer;
import java.io.IOException;

import org.w3c.dom.*;


/**
 * Node representing XML text.
 *
 * <P> Subclasses should not currently attempt to modify the
 * representation of content, but may add new methods to support
 * more sophisticated access or manipulation of that content.
 *
 * @author David Brownell
 * @version $Revision: 1.4 $
 */
// public
class TextNode extends DataNode implements Text
{
    //
    // XXX Really want a more flexible representation of text than
    // we have here ... it should be possible to take up only part
    // of a buffer, and in fact to be able to share one with other
    // text nodes.  For readonly trees that will reduce the heap
    // impact; and for editable ones it can facilitate intelligent
    // editing support.  Text seems more expensive than attributes,
    // which is quite odd.
    //

    /**
     * Constructs a text object with no text and unattached
     * to any document.
     */
    public TextNode () { }

        
    /**
     * Constructs text object by copying text from the input buffer.
     */
    public TextNode (char buf [], int offset, int len)
    {
	super (buf, offset, len);
    }

    /**
     * Constructs a text object by copying text from the string.
     */
    public TextNode (String s)
    {
	super (s);
    }

    /**
     * Writes the text, escaping XML metacharacters as needed
     * to let this text be parsed again without change.
     */
    public void writeXml (XmlWriteContext context) throws IOException
    {
	Writer	out = context.getWriter ();
	int	start = 0, last = 0;

	// XXX saw this once -- being paranoid
	if (data == null)
	    { System.err.println ("Null text data??"); return; }

	while (last < data.length) {
	    char c = data [last];

	    //
	    // escape markup delimiters only ... and do bulk
	    // writes wherever possible, for best performance
	    //
	    // note that character data can't have the CDATA
	    // termination "]]>"; escaping ">" suffices, and
	    // doing it very generally helps simple parsers
	    // that may not be quite correct.
	    //
	    if (c == '<') {			// not legal in char data
		out.write (data, start, last - start);
		start = last + 1;
		out.write ("&lt;");
	    } else if (c == '>') {		// see above
		out.write (data, start, last - start);
		start = last + 1;
		out.write ("&gt;");
	    } else if (c == '&') {		// not legal in char data
		out.write (data, start, last - start);
		start = last + 1;
		out.write ("&amp;");
	    }
	    last++;
	}
	out.write (data, start, last - start);
    }

    /**
     * Combines this text node with its next sibling to create a
     * single text node.  If the next node is not text, nothing is
     * done.  This should be used with care, since large spans of
     * text may not be efficient to represent.
     */
    public void joinNextText ()
    {
	Node	next = getNextSibling ();
	char	tmp [], nextText [];

	if (next == null || next.getNodeType () != TEXT_NODE)
	    return;
	getParentNode ().removeChild (next);

	nextText = ((TextNode)next).getText ();
	tmp = new char [data.length + nextText.length];
	System.arraycopy (data, 0, tmp, 0, data.length);
	System.arraycopy (nextText, 0, tmp, data.length, nextText.length);
	data = tmp;
    }


    // DOM support

    /** DOM: Returns the TEXT_NODE node type constant. */
    public short getNodeType () { return TEXT_NODE; }


    /**
     * DOM:  Splits this text node into two, returning the part
     * beginning at <em>offset</em>.  The original node has that 
     * text removed, and the two nodes are siblings in the natural
     * order.
     */
    public Text splitText (int offset)
    throws DOMException
    {
	TextNode	retval;
	char		delta [];

        if (isReadonly ())
	    throw new DomEx (DomEx.NO_MODIFICATION_ALLOWED_ERR);
	
	try {
		retval = new TextNode (data, offset, data.length - offset);
	}
	catch (ArrayIndexOutOfBoundsException ae) {
		throw new DomEx (DOMException.INDEX_SIZE_ERR);
	}
	catch (NegativeArraySizeException nae) {
		throw new DomEx (DOMException.INDEX_SIZE_ERR);
	}
	getParentNode().insertBefore (retval, getNextSibling ());
	delta = new char [offset];
	System.arraycopy (data, 0, delta, 0, offset);
	data = delta;
	return retval;
    }


    /**
     * DOM: returns a new text node with the same contents as this one.
     */
    public Node cloneNode (boolean deep)
    {
    	TextNode retval = new TextNode (data, 0, data.length);
	retval.setOwnerDocument ((XmlDocument) this.getOwnerDocument ());
	return retval;
    }

    /**
     * DOM:  Returns the string "#text".
     */
    public String getNodeName () { return "#text"; }
}
