/*
 * $Id: SAXParserFactory.java,v 1.15.2.1 2000/04/13 20:42:35 mode Exp $
 * 
 * Copyright (c) 1998-1999 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 */


package javax.xml.parsers;

import org.xml.sax.Parser;
import org.xml.sax.SAXException;

import java.io.InputStream;
import java.io.IOException;
import java.io.File;
import java.io.FileInputStream;
import java.util.Locale;

/**
 * Defines a factory API that enables applications to configure and
 * obtain a SAX based parser to parse XML documents. 
 *
 * @since JAXP 1.0
 * @version 1.0
 */

public abstract class SAXParserFactory {

    private boolean validating = false;
    private boolean namespaceAware= false;
    
    protected SAXParserFactory () {
    
    }

    /**
     * Obtain a new instance of a <code>SAXParserFactory</code>. This
     * static method creates a new factory instance based on a System
     * property setting or uses the platform default if no property
     * has been defined.<p>
     *
     * The system property that controls which Factory implementation
     * to create is named
     * &quot;javax.xml.parsers.SAXParserFactory&quot;. This property
     * names a class that is a concrete subclass of this abstract
     * class. If no property is defined, a platform default will be
     * used.<p>
     *
     * Once an application has obtained a reference to a
     * <code>SAXParserFactory</code> it can use the factory to
     * configure and obtain parser instances.
     *
     * @exception FactoryConfigurationError if the implementation is
     * not available or cannot be instantiated.
     */

    public static SAXParserFactory newInstance() {
	String factoryImplName = null;
        try {
            factoryImplName =
                System.getProperty("javax.xml.parsers.SAXParserFactory", 
                               "com.sun.xml.parser.SAXParserFactoryImpl");
	}catch (SecurityException se) {
	    factoryImplName = "com.sun.xml.parser.SAXParserFactoryImpl";	
	}
        SAXParserFactory factoryImpl;
        try {
            Class clazz = Class.forName(factoryImplName);
            factoryImpl = (SAXParserFactory)clazz.newInstance();
        } catch  (ClassNotFoundException cnfe) {
	    throw new FactoryConfigurationError(cnfe);
	} catch (IllegalAccessException iae) {
	    throw new FactoryConfigurationError(iae);
	} catch (InstantiationException ie) {
	    throw new FactoryConfigurationError(ie);
	}
        return factoryImpl;
    }

    /**
     * Creates a new instance of a SAXParser using the currently
     * configured factory parameters.
     *
     * @exception ParserConfigurationException if a parser cannot
     * be created which satisfies the requested configuration.
     */
    
    public abstract SAXParser newSAXParser()
        throws ParserConfigurationException, SAXException;

    
    /**
     * Specifies that the parser produced by this code will
     * provide support for XML namespaces.
     */
    
    public void setNamespaceAware(boolean awareness) 
    {
        this.namespaceAware = awareness;
    }

    /**
     * Specifies that the parser produced by this code will
     * validate documents as they are parsed.
     */
    
    public void setValidating(boolean validating) 
    {
        this.validating = validating;
    }

    /**
     * Indicates whether or not the factory is configured to produce
     * parsers which are namespace aware.
     */
    
    public boolean isNamespaceAware() {
        return namespaceAware;
    }

    /**
     * Indicates whether or not the factory is configured to produce
     * parsers which validate the XML content during parse.
     */
    
    public boolean isValidating() {
        return validating;
    }

}

