/*
 * @(#)IIOMetadata.java	1.22 00/10/19
 *
 * Copyright 2000 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

package javax.imageio.metadata;

import org.w3c.dom.Node;

/**
 * An abstract class to be extended by objects that represent metadata
 * (non-image data) associated with images and streams.  Plug-ins
 * represent metadata using opaque, plug-in specific objects.  These
 * objects, however, provide the ability to access their internal
 * information as a tree of <code>IIOMetadataNode</code> objects that
 * support the XML DOM interfaces as well as additional interfaces for
 * storing non-textual data and retrieving information about legal
 * data values.  The format of such trees is plug-in dependent, but
 * plug-ins may choose to support a plug-in neutral format described
 * below.  A single plug-in may support multiple metadata formats,
 * whose names maybe determined by calling
 * <code>getMetadataFormatNames</code>.  The plug-in may also support
 * a single special format, referred to as the "native" format, which
 * is designed to encode its metadata losslessly.  This format will
 * typically be designed specifically to work with a specific file
 * format, so that images may be loaded and saved in the same format
 * with no loss of metadata, but may be less useful for transcoding
 * between different formats.
 *
 * @see javax.imageio.ImageReader#getImageMetadata
 * @see javax.imageio.ImageReader#getStreamMetadata
 * @see javax.imageio.ImageReader#readAll
 * @see javax.imageio.ImageWriter#getDefaultStreamMetadata
 * @see javax.imageio.ImageWriter#getDefaultImageMetadata
 * @see javax.imageio.ImageWriter#write
 * @see javax.imageio.IIOImage
 * @see javax.imageio.ImageTranscoder
 *
 * @version 0.5
 */
public abstract class IIOMetadata {

    /**
     * The name of the native metadata format for this object,
     * initialized to <code>null</code> and set via the constructor.
     */
    protected String nativeMetadataFormatName = null;

    /**
     * The names of the metadata formats supported by this object, as
     * an array of <code>String</code>s, initialized to
     * <code>null</code> and set via the constructor.
     */
    protected String[] metadataFormatNames = null;

    /**
     * The default <code>IIOMetadataController</code> that will be
     * used to provide settings for this <code>IIOMetadata</code>
     * object when the <code>activateController</code> method
     * is called.  This default should be set by subclasses
     * that choose to provide their own default controller, 
     * usually a GUI, for setting parameters.
     *
     * @see IIOMetadataController
     * @see #getDefaultController
     * @see #activateController
     */
    protected IIOMetadataController defaultController = null;

    /**
     * The <code>IIOMetadataController</code> that will be
     * used to provide settings for this <code>IIOMetadata</code>
     * object when the <code>activateController</code> method
     * is called.  This value overrides any default controller, 
     * even when <code>null</code>.
     *
     * @see IIOMetadataController
     * @see #setController(IIOMetadataController)
     * @see #hasController()
     * @see #activateController()
     */
    protected IIOMetadataController controller = null;

    /**
     * Constructs an empty <code>IIOMetadata</code> object.
     */
    protected IIOMetadata() {}

    /**
     * Constructs an <code>IIOMetadata</code> object with a given
     * native metadata format name, and a set of supported metadata
     * formats (including the native format, if any).
     *
     * @param nativeMetadataFormatName the name of the native metadata
     * format, as a <code>String</code>.
     * @param metadataFormatNames an array of <code>String</code>s
     * indicating the formats supported by this object.  The native
     * format name should be the first element of the array.
     */
    protected IIOMetadata(String nativeMetadataFormatName,
                          String[] metadataFormatNames) {
        this.nativeMetadataFormatName = nativeMetadataFormatName;
        if (metadataFormatNames != null) {
            this.metadataFormatNames = (String[])metadataFormatNames.clone();
        }
    }

    /**
     * Returns <code>true</code> if this object does not support the
     * <code>mergeTree</code>, <code>setFromTree</code>, and
     * <code>reset</code> methods.
     *
     * @return true if this <code>IIOMetadata</code> object cannot be
     * modified.
     */
    public abstract boolean isReadOnly();

    /**
     * Returns an array of <code>String</code>s containing the names
     * of the metadata formats recognized by this plug-in's
     * <code>getAsTree</code>, <code>setFromTree</code>, and
     * <code>mergeTree</code> methods.
     *
     * <p> There should always be at least one value returned; if a
     * plug-in wishes not to support metadata, it should return
     * <code>null</code> wherever an <code>IIOMetadata</code>object
     * would normally be returned.
     *
     * <p> The formats should be returned roughly in order of
     * decreasing fidelity, with any "native" formats providing
     * lossless encoding listed first.  The "native" format, if any,
     * may be determined by calling
     * <code>getNativeMetadataFormatName</code>.
     *
     * @return an array of <code>String</code>s.
     *
     * @see #getAsTree
     * @see #setFromTree
     * @see #mergeTree
     * @see #getNativeMetadataFormatName
     */
    public String[] getMetadataFormatNames() {
        if (metadataFormatNames == null) {
            return null;
        }
        return (String[])metadataFormatNames.clone();
    }

    /**
     * Returns the name of the "native" metadata format for this
     * plug-in, which typically allows for lossless encoding and
     * transmission of the metadata stored in the format handled by
     * this plug-in.  If no such format is supported,
     * <code>null</code>will be returned.  If a
     * non-<code>null</code> value is returned, it will be one of the
     * names returned by <code>getMetadataFormatNames</code>.
     *
     * <p> The structure and contents of the "native" metadata format
     * are defined by the plug-in that created this
     * <code>IIOMetadata</code> object.  Plug-ins for simple formats
     * will usually create a dummy node for the root, and then a
     * series of child nodes representing individual tags, chunks, or
     * keyword/value pairs.  A plug-in may choose whether or not to
     * document its native format.
     *
     * @return the name of the native format, or <code>null</code>.
     *
     * @see #getMetadataFormatNames
     */
    public String getNativeMetadataFormatName() {
        return nativeMetadataFormatName;
    }

    /**
     * Returns an <code>IIOMetadataFormat</code> object describing the
     * given metadata format, or <code>null</code> if no description
     * is available.  The supplied name must be among those returned
     * by <code>getMetadataFormatNames</code>.
     *
     * @param formatName the desired metadata format.
     *
     * @return an <code>IIOMetadataFormat</code> object.
     *
     * @exception IllegalArgumentException if <code>formatName</code>
     * is <code>null</code> or is not one of the names returned by
     * <code>getMetadataFormatNames</code>.
     */
    public abstract IIOMetadataFormat getMetadataFormat(String formatName);

    /**
     * Returns an XML DOM <code>Node</code> object that represents the
     * root of a tree of metadata contained within this object
     * according to the conventions defined by a given metadata
     * format.
     *
     * <p> The names of other available metadata formats may be
     * queried using the <code>getMetadataFormatNames</code> method.
     *
     * @param formatName the desired metadata format.
     *
     * @return an XML DOM <code>Node</code> object forming the
     * root of a tree.
     *
     * @exception IllegalArgumentException if <code>formatName</code>
     * is <code>null</code> or is not one of the names returned by
     * <code>getMetadataFormatNames</code>.
     * 
     * @see #getMetadataFormatNames
     * @see #setFromTree
     * @see #mergeTree
     */
    public abstract Node getAsTree(String formatName);

    /**
     * Alters the internal state of this <code>IIOMetadata</code>
     * object from a tree of XML DOM <code>Node</code>s whose syntax
     * is defined by the given metadata format.  The previous state is
     * altered only as necessary to accomodate the nodes that are
     * present in the given tree.  If the tree structure or contents
     * are invalid, an <code>IIOInvalidTreeException</code> will be
     * thrown.
     *
     * @param formatName the desired metadata format.
     * @param root an XML DOM <code>Node</code> object forming the
     * root of a tree.
     *
     * @exception IllegalStateException if this object is read-only.
     * @exception IllegalArgumentException if <code>formatName</code>
     * is <code>null</code> or is not one of the names returned by
     * <code>getMetadataFormatNames</code>.
     * @exception IllegalArgumentException if <code>root</code> is
     * <code>null</code>.
     * @exception IIOInvalidTreeException if the tree cannot be parsed
     * successfully using the rules of the given format.
     *
     * @see #getMetadataFormatNames
     * @see #getAsTree
     * @see #setFromTree
     */
    public abstract void mergeTree(String formatName, Node root)
        throws IIOInvalidTreeException;

    /**
     * Sets the internal state of this <code>IIOMetadata</code> object
     * from a tree of XML DOM <code>Node</code>s whose syntax is
     * defined by the given metadata format.  The previous state is
     * discarded.  If the tree's structure or contents are invalid, an
     * <code>IIOInvalidTreeException</code> will be thrown.
     *
     * <p> The default implementation calls <code>reset</code>
     * followed by <code>mergeTree(formatName, root)</code>.
     *
     * @param formatName the desired metadata format.
     * @param root an XML DOM <code>Node</code> object forming the
     * root of a tree.
     *
     * @exception IllegalStateException if this object is read-only.
     * @exception IllegalArgumentException if <code>formatName</code>
     * is <code>null</code> or is not one of the names returned by
     * <code>getMetadataFormatNames</code>.
     * @exception IllegalArgumentException if <code>root</code> is
     * <code>null</code>.
     * @exception IIOInvalidTreeException if the tree cannot be parsed
     * successfully using the rules of the given format.
     *
     * @see #getMetadataFormatNames
     * @see #getAsTree
     * @see #mergeTree
     */
    public void setFromTree(String formatName, Node root)
        throws IIOInvalidTreeException {
        reset();
        mergeTree(formatName, root);
    }

    /**
     * Resets all the data stored in this object to default values.
     * The combination of <code>reset</code> and
     * <code>mergeTree</code> should be equivalent to calling
     * <code>setFromTree</code>.
     *
     * @exception IllegalStateException if this object is read-only.
     */
    public abstract void reset();

    /**
     * Sets the <code>IIOMetadataController</code> to be used
     * to provide settings for this <code>IIOMetadata</code>
     * object when the <code>activateController</code> method
     * is called, overriding any default controller.  If the
     * argument is <code>null</code>, no controller will be
     * used, including any default.  To restore the default, use
     * <code>setController(getDefaultController())</code>.
     *
     * @param controller An appropriate 
     * <code>IIOMetadataController</code>, or <code>null</code>.
     *
     * @see IIOMetadataController
     * @see #getController
     * @see #getDefaultController
     * @see #hasController
     * @see #activateController()
     */
    public void setController(IIOMetadataController controller) {
        this.controller = controller;
    }

    /**
     * Returns whatever <code>IIOMetadataController</code> is currently
     * installed.  This could be the default if there is one, 
     * <code>null</code>, or the argument of the most recent call
     * to <code>setController</code>.
     *
     * @return the currently installed
     * <code>IIOMetadataController</code>, or <code>null</code>.
     *
     * @see IIOMetadataController
     * @see #setController
     * @see #getDefaultController
     * @see #hasController
     * @see #activateController()
     */
    public IIOMetadataController getController() {
        return controller;
    }

    /**
     * Returns the default <code>IIOMetadataController</code>, if there
     * is one, regardless of the currently installed controller.  If
     * there is no default controller, returns <code>null</code>.
     *
     * @return the default <code>IIOMetadataController</code>, or
     * <code>null</code>.
     *
     * @see IIOMetadataController
     * @see #setController(IIOMetadataController)
     * @see #getController
     * @see #hasController
     * @see #activateController()
     */
    public IIOMetadataController getDefaultController() {
        return defaultController;
    }

    /**
     * Returns <code>true</code> if there is a controller installed
     * for this <code>IIOMetadata</code> object.  This will return
     * <code>true</code> if <code>getController</code> would not
     * return <code>null</code>.
     *
     * @return <code>true</code> if a controller is installed.
     *
     * @see IIOMetadataController
     * @see #setController(IIOMetadataController)
     * @see #getController
     * @see #getDefaultController
     * @see #activateController()
     */
    public boolean hasController() {
        return (controller != null);
    }

    /**
     * Activates the installed <code>IIOMetadataController</code> for
     * this <code>IIOMetadata</code> object.  When this method returns,
     * all values for this <code>IIOMetadata</code> object will be
     * ready for the next IO operation.  Ordinarily, the controller
     * will be a GUI providing a user interface for a subclass of
     * <code>IIOMetadata</code> for a particular plug-in.  Controllers
     * need not be GUIs, however.
     *
     * @exception IllegalStateException if there is no no controller
     * currently installed.
     *
     * @see IIOMetadataController
     * @see #setController(IIOMetadataController)
     * @see #getController
     * @see #getDefaultController
     * @see #hasController
     */
    public void activateController() {
        if (controller == null) {
            throw new IllegalStateException();
        }
        controller.activate(this);
    }
}
