/*
 * @(#)ImageReaderSpi.java	1.34 00/11/07
 *
 * Copyright 2000 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

package javax.imageio.spi;

import java.io.IOException;
import javax.imageio.ImageReader;
import javax.imageio.IIOException;
import javax.imageio.stream.ImageInputStream;

/**
 * The service provider interface (SPI) for <code>ImageReader</code>s.
 * For more information on service provider classes, see the class comment
 * for the <code>IIORegistry</code> class.
 * 
 * <p> Each <code>ImageReaderSpi</code> provides several types of information
 * about the <code>ImageReader</code> class with which it is associated.
 *
 * <p> The name of the vendor who defined the SPI class and a
 * brief description of the class are available via the
 * <code>getVendorName</code>, <code>getDescription</code>,
 * and <code>getVersion</code> methods.
 * These methods may be internationalized to provide locale-specific
 * output.  These methods are intended mainly to provide short,
 * human-readable information that might be used to organize a pop-up
 * menu or other list.
 *
 * <p> Lists of format names, file suffixes, and MIME types associated
 * with the service may be obtained by means of the
 * <code>getFormatNames</code>, <code>getFileSuffixes</code>, and
 * <code>getMIMETypes</code> methods.  These methods may be used to
 * identify candidate <code>ImageReader</code>s for decoding a
 * particular file or stream based on manual format selection, file
 * naming, or MIME associations (for example, when accessing a file
 * over HTTP or as an email attachment).
 *
 * <p> A more reliable way to determine which <code>ImageReader</code>s
 * are likely to be able to parse a particular data stream is provided
 * by the <code>canDecodeInput</code> method.  This methods allows the
 * service provider to inspect the actual stream contents.
 *
 * <p> Finally, an instance of the <code>ImageReader</code> class
 * associated with this service provider may be obtained by calling
 * the <code>createReaderInstance</code> method.  Any heavyweight
 * initialization, such as the loading of native libraries or creation
 * of large tables, should be deferred at least until the first
 * invocation of this method.
 *
 * @see IIORegistry
 * @see javax.imageio.ImageReader
 *
 * @version 0.5
 */
public abstract class ImageReaderSpi extends ImageReaderWriterSpi {

    /**
     * A single-element array containing
     * <code>ImageInputStream.class</code>, to be returned from
     * <code>getInputTypes</code>.
     */
    public static final Class[] STANDARD_INPUT_TYPE =
        { ImageInputStream.class };

    /**
     * An array of <code>Class</code> objects to be returned from
     * <code>getInputTypes</code>.
     */
    protected Class[] inputTypes;

    /**
     * An array of strings to be returned from
     * <code>getImageWriterSpiNames</code>.
     */
    protected String[] writerSpiNames;

    /**
     * The <code>Class</code> of the reader, initially
     * <code>null</code>.
     */
    private Class readerClass = null;
    
    /**
     * Constructs a blank <code>ImageReaderSpi</code>.  It is up to
     * the subclass to initialize instance variables and/or override
     * method implementations in order to provide working versions of
     * all methods.
     */
    protected ImageReaderSpi() {
    }

    /**
     * Constructs an <code>ImageReaderSpi</code> with a given
     * set of values.
     *
     * @param vendorName the vendor name.
     * @param version a version identifier.
     * @param names an array of <code>String</code>s indicating the
     * format names.
     * @param suffixes an array of <code>String</code>s indicating the
     * common file suffixes.
     * @param MIMETypes an array of <code>String</code>s indicating
     * the format's MIME types.
     * @param inputTypes and array of <code>Class</code> objects indicating
     * the legal input types.
     * @param readerClassName the fully-qualified name of the
     * associated <code>ImageReader</code> class.
     * @param writerSpiNames an array of <code>String</code>s naming
     * the classes of all associated <code>ImageWriter</code>s.
     * @param streamMetadataFormatNames an array of
     * <code>String</code>s, or <code>null</code>, to be returned from
     * <code>getStreamMetadataFormatNames</code>.
     * @param nativeStreamMetadataFormatName a
     * <code>String</code>, or <code>null</code>, to be returned from
     * <code>getNativeStreamMetadataFormatName</code>.
     * @param imageMetadataFormatNames an array of
     * <code>String</code>s to be returned from
     * <code>getImageMetadataFormatNames</code>.
     * @param nativeImageMetadataFormatName a
     * <code>String</code>, or <code>null</code>, to be returned from
     * <code>getNativeImageMetadataFormatName</code>.
     */
    public ImageReaderSpi(String vendorName,
                          String version,
                          String[] names,
                          String[] suffixes,
                          String[] MIMETypes,
                          String readerClassName,
                          Class[] inputTypes,
                          String[] writerSpiNames,
                          String[] streamMetadataFormatNames,
                          String nativeStreamMetadataFormatName,
                          String[] imageMetadataFormatNames,
                          String nativeImageMetadataFormatName) {
        super(vendorName, version,
              names, suffixes, MIMETypes, readerClassName,
              streamMetadataFormatNames,
              nativeStreamMetadataFormatName,
              imageMetadataFormatNames,
              nativeImageMetadataFormatName);
        this.inputTypes = (Class[])inputTypes.clone();
        this.writerSpiNames = (String[])writerSpiNames.clone();
    }

    /**
     * Returns an array of <code>Class</code> objects indicating what
     * types of objects may be used as arguments to the reader's
     * <code>setInput</code> method.
     *
     * <p> For most readers, which only accept input from an
     * <code>ImageInputStream</code>, a single-element array
     * containing <code>ImageInputStream</code>.class should be
     * returned.
     *
     * @return an array of <code>Class</code>objects.
     */
    public Class[] getInputTypes() {
        return (Class[])inputTypes.clone();
    }

    /**
     * Returns <code>true</code> if the supplied source object appears
     * to be of the format supported by this reader.  Returning
     * <code>true</code> from this method does not guarantee that
     * reading will succeed, only that there appears to be a
     * reasonable chance of success based on a brief inspection of the
     * stream contents.  If the source is an
     * <code>ImageInputStream</code>, implementations will commonly
     * check the first several bytes of the stream for a "magic
     * number" associated with the format.  Once actual reading has
     * commenced, the reader may still indicate failure at any time
     * prior to the completion of decoding.
     *
     * <p> It is important that the state of the object not be
     * disturbed in order that other <code>ImageReaderSpi</code>s can
     * properly determine whether they are able to decode the object.
     * In particular, if the source is an
     * <code>ImageInputStream</code>, a
     * <code>mark</code>/<code>reset</code> pair should be used to
     * preserve the stream position.
     *
     * <p> Formats such as "raw," which can potentially attempt
     * to read nearly any stream, should return <code>false</code>
     * in order to avoid being invoked in preference to a closer
     * match.
     *
     * <p> If <code>source</code> is not an instance of one of the
     * classes returned by <code>getInputTypes</code>, the method
     * should simply return <code>false</code>.
     *
     * @param source the object (typically an
     * <code>ImageInputStream</code>) to be decoded.
     *
     * @return <code>true</code> if it is likely that this stream can
     * be decoded.
     *
     * @exception IllegalArgumentException if <code>source</code> is
     * <code>null</code>.
     * @exception IOException if an I/O error occurs while reading the
     * stream.
     */
    public abstract boolean canDecodeInput(Object source) throws IOException;

    /**
     * Returns an instance of the <code>ImageReader</code>
     * implementation associated with this service provider.
     * The returned object will initially be in an initial state
     * as if its <code>reset</code> method had been called.
     *
     * <p> The default implementation simply returns
     * <code>createReaderInstance(null)</code>.
     *
     * @return an <code>ImageReader</code> instance.
     *
     * @exception IIOException if an error occurs during loading, 
     * or initialization of the reader class, or during instantiation
     * or initialization of the reader object.
     */
    public ImageReader createReaderInstance() throws IIOException {
        return createReaderInstance(null);
    }

    /**
     * Returns an instance of the <code>ImageReader</code>
     * implementation associated with this service provider.
     * The returned object will initially be in an initial state
     * as if its <code>reset</code> method had been called.
     *
     * <p> An <code>Object</code> may be supplied to the plug-in at
     * construction time.  The nature of the object is entirely
     * plug-in specific.
     *
     * <p> Typically, a plug-in will implement this method using code
     * such as <code>return new MyImageReader(this)</code>.
     *
     * @param extension a plug-in specific extension object, which may
     * be <code>null</code>.
     *
     * @return an <code>ImageReader</code> instance.
     *
     * @exception IIOException if the attempt to instantiate
     * the reader fails.
     * @exception IllegalArgumentException if the
     * <code>ImageReader</code>'s contructor throws an
     * <code>IllegalArgumentException</code> to indicate that the
     * extension object is unsuitable.
     */
    public abstract ImageReader createReaderInstance(Object extension) 
        throws IIOException;

    /**
     * Returns <code>true</code> if the <code>ImageReader</code> object
     * passed in is an instance of the <code>ImageReader</code> 
     * associated with this service provider.
     *
     * <p> The default implementation compares the fully-qualified
     * class name of the <code>reader</code> argument with the class
     * name passed into the constructor.  This method may be overridden
     * if more sophisticated checking is required.
     *
     * @param reader an <code>ImageReader</code> instance.
     *
     * @return <code>true</code> if <code>reader</code> is recognized.
     *
     * @exception IllegalArgumentException if <code>reader</code> is
     * <code>null</code>.
     */
    public boolean isOwnReader(ImageReader reader) {
        if (reader == null) {
            throw new IllegalArgumentException();
        }
        String name = reader.getClass().getName();
        return name.equals(pluginClassName);
    }

    /**
     * Returns an array of <code>String</code>s containing the fully
     * qualified
     * names of all the <code>ImageWriterSpi</code> classes that can
     * understand the internal metadata representation used by the
     * <code>ImageReader</code> associated with this service provider,
     * or <code>null</code> if there are no such <code>ImageWriter</code>s
     * specified.
     *
     * <p> The first item in the array must be the name of the service
     * provider for the "preferred" writer, as it will be used to
     * instantiate the <code>ImageWriter</code> returned by
     * <code>ImageIO.getImageWriter(ImageReader)</code>.
     *
     * <p> This mechanism may be used to obtain
     * <code>ImageWriters</code> that will understand the internal
     * structure of non-pixel meta-data (see
     * <code>IIOTreeInfo</code>) generated by an
     * <code>ImageReader</code>.  By obtaining this data from the
     * <code>ImageReader</code> and passing it on to one of the
     * <code>ImageWriters</code> obtained with this method, a client
     * program can read an image, modify it in some way, and write it
     * back out while preserving all meta-data, without having to
     * understand anything about the internal structure of the
     * meta-data, or even about the image format.
     *
     * @return an array of Strings containing names of 
     * <code>ImageWriterSpi</code>, or <code>null</code>.
     *
     * @see javax.imageio.ImageIO#getImageWriter(ImageReader)
     */
    public String[] getImageWriterSpiNames() {
        return ((String[])writerSpiNames.clone());
    }
}
