/*
 * @(#)ImageInputStreamImpl.java	1.25 00/10/20
 *
 * Copyright 2000 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

package javax.imageio.stream;

import java.io.DataInput;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Stack;

/**
 * An abstract class implementing the <code>ImageInputStream</code> interface.
 * This class is designed to reduce the number of methods that must
 * be implemented by subclasses.
 *
 * <p> In particular, this class handles most or all of the details of
 * byte order interpretation, buffering, mark/reset, discarding,
 * closing, and disposing.
 *
 * @version 0.5
 */
public abstract class ImageInputStreamImpl implements ImageInputStream {

    private static final int DEFAULT_BUFFER_SIZE = 1048576;

    private Stack markByteStack = new Stack();

    private Stack markBitStack = new Stack();

    private boolean isClosed = false;

//      /**
//       * A <code>File</code> that may be used to buffer data, or
//       * <code>null</code>.
//       */
//      protected File cacheFile;

    /**
     * The byte order of the stream as a <code>boolean</code>, where
     * true <code>indicates</code> network byte order and false
     * indicates the reverse order.  By default, the value is
     * <code>true</code>.
     */
    protected boolean networkByteOrder = true;

    /**
     * The current read position within the stream.  Subclasses are
     * responsible for keeping this value current from any method they
     * override that alters the read position.
     */
    protected long streamPos;

    /**
     * The current bit offset within the stream.  Subclasses are
     * responsible for keeping this value current from any method they
     * override that alters the bit offset.
     */
    protected int bitOffset;

    /**
     * The position prior to which data may be discarded.  Seeking
     * to a smaller position is not allowed.  <code>flushedPos</code>
     * will always be >= 0.
     */
    protected long flushedPos = 0;

//      /**
//       * Constructs an <code>ImageInputStreamImpl</code> and
//       * optionally sets a <code>File</code> to be used as a cache for
//       * previously read data and for readahead.
//       *
//       * @param cacheFile a <code>File</code> that may be used to buffer
//       * data, or <code>null</code>.
//       */
//      public ImageInputStreamImpl(File cacheFile) {
//          this.cacheFile = cacheFile;
//      }

    /**
     * Constructs an <code>ImageInputStreamImpl</code>.
     */
    public ImageInputStreamImpl() {
    }

    /**
     * Throws an <code>IOException</code> if the stream has been closed.
     * Subclasses may call this method from any of their methods that
     * require the stream not to be closed.
     *
     * @exception IOException if the stream is closed.
     */
    protected final void checkClosed() throws IOException {
        if (isClosed) {
            throw new IOException("closed");
        }
    }

    public void setByteOrder(boolean networkByteOrder) {
        this.networkByteOrder = networkByteOrder;
    }

    public boolean getByteOrder() {
        return networkByteOrder;
    }

    /*
     * Reads a single byte from the stream and returns it as an
     * <code>int</code> between 0 and 255.  If EOF is reached, 
     * <code>-1</code> is returned.
     *
     * <p> Subclasses must provide an implementation for this method.
     * The subclass implementation should update the stream position
     * before exiting.
     *
     * <p> The bit offset within the stream must be reset to zero before
     * the read occurs.
     *
     * @exception IIOException if the stream has been closed.
     */
    public abstract int read() throws IOException;

    /**
     * A convenience method that calls <code>read(b, 0, b.length)</code>.
     *
     * <p> The bit offset within the stream is reset to zero before
     * the read occurs.
     *
     */
    public int read(byte[] b) throws IOException {
        return read(b, 0, b.length);
    }

    /**
     * Reads one or more bytes from the stream, up to
     * <code>len</code>, and stores them into <code>b</code> starting
     * at index <code>off</code>.  The number of bytes read is
     * returned.
     *
     * <p> Subclasses must provide an implementation for this method.
     * The subclass implementation should update the stream position
     * before exiting. 
     *
     * <p> The bit offset within the stream must be reset to zero before
     * the read occurs.
     *
     * @param b an array of bytes to be written to.
     * @param off the starting position within <code>b</code> to write to.
     * @param len the maximum number of bytes to read.
     *
     * @return the number of bytes actually read.
     *
     * @exception IndexOutOfBoundsException if <code>off</code> is
     * negative, <code>len</code> is negative, or <code>off +
     * len</code> is greater than <code>b.length</code>.
     * @exception NullPointerException if <code>b</code> is
     * <code>null</code>.
     * @exception IOException if an I/O error occurs.
     */
    public abstract int read(byte[] b, int off, int len) throws IOException;

    public void readBytes(IIOByteBuffer buf, int len) throws IOException {
        byte[] data = new byte[len];
        len = read(data, 0, len);
        
        buf.setData(data);
        buf.setOffset(0);
        buf.setLength(len);
    }

    public boolean readBoolean() throws IOException {
	int ch = this.read();
	if (ch < 0)
	    throw new EOFException();
	return (ch != 0);
    }

    public byte readByte() throws IOException {
	int ch = this.read();
	if (ch < 0)
	    throw new EOFException();
	return (byte)ch;
    }

    public int readUnsignedByte() throws IOException {
        int ch = this.read();
        if (ch < 0)
    	    throw new EOFException();
        return ch;
    }

    public short readShort() throws IOException {
	int ch1 = this.read();
	int ch2 = this.read();
	if ((ch1 | ch2) < 0)
	    throw new EOFException();

        if (networkByteOrder) {
            return (short)((ch1 << 8) + (ch2 << 0));
        } else {
            return (short)((ch2 << 8) + (ch1 << 0));
        }
    }

    public int readUnsignedShort() throws IOException {
        return ((int)readShort()) & 0xffff;
    }

    public char readChar() throws IOException {
        return (char)readShort();
    }

    public int readInt() throws IOException {
	int ch1 = this.read();
	int ch2 = this.read();
	int ch3 = this.read();
	int ch4 = this.read();
	if ((ch1 | ch2 | ch3 | ch4) < 0)
	    throw new EOFException();

        if (networkByteOrder) {
            return ((ch1 << 24) + (ch2 << 16) + (ch3 << 8) + (ch4 << 0));
        } else {
            return ((ch4 << 24) + (ch3 << 16) + (ch2 << 8) + (ch1 << 0));
        }
    }

    public long readUnsignedInt() throws IOException {
        return ((long)readInt()) & 0xffffffff;
    }

    public long readLong() throws IOException {
        int i1 = readInt();
        int i2 = readInt();

        if (networkByteOrder) {
            return ((long)i1 << 32) + (i2 & 0xFFFFFFFFL);
        } else {
            return ((long)i2 << 32) + (i1 & 0xFFFFFFFFL);
        }
    }

    public float readFloat() throws IOException {
        return Float.intBitsToFloat(readInt());
    }

    public double readDouble() throws IOException {
	return Double.longBitsToDouble(readLong());
    }

    public String readLine() throws IOException {
	StringBuffer input = new StringBuffer();
	int c = -1;
	boolean eol = false;

	while (!eol) {
	    switch (c = read()) {
	    case -1:
	    case '\n':
		eol = true;
		break;
	    case '\r':
		eol = true;
		long cur = getStreamPosition();
		if ((read()) != '\n') {
		    seek(cur);
		}
		break;
	    default:
		input.append((char)c);
		break;
	    }
	}

	if ((c == -1) && (input.length() == 0)) {
	    return null;
	}
	return input.toString();
    }

    public String readUTF() throws IOException {
        this.bitOffset = 0;
	return DataInputStream.readUTF(this);
    }

    public void readFully(byte[] b, int off, int len) throws IOException {
        if (len < 0) {
            throw new IndexOutOfBoundsException();
        }

        while (len > 0) {
            int nbytes = read(b, off, len);
            if (nbytes == -1) {
                throw new EOFException();
            }
            off += nbytes;
            len -= nbytes;
        }
    }

    public void readFully(byte[] b) throws IOException {
        readFully(b, 0, b.length);
    }

    public void readFully(short[] s, int off, int len) throws IOException {
        if (len < 0 || off + len > s.length) {
            throw new IllegalArgumentException();
        }

        byte[] b = new byte[2*len];
        readFully(b, 0, 2*len);

        int boff = 0;
        if (networkByteOrder) {
            for (int j = 0; j < len; j++) {
                int b0 = b[boff];
                int b1 = b[boff + 1];
                s[off + j] = (short)((b0 << 8) | b1);
                boff += 2;
            }
        } else {
            for (int j = 0; j < len; j++) {
                int b0 = b[boff];
                int b1 = b[boff + 1];
                s[off + j] = (short)((b1 << 8) | b0);
                boff += 2;
            }
        }
    }

    public void readFully(char[] c, int off, int len) throws IOException {
        if (len < 0 || off + len > c.length) {
            throw new IllegalArgumentException();
        }

        byte[] b = new byte[2*len];
        readFully(b, 0, 2*len);

        int boff = 0;
        if (networkByteOrder) {
            for (int j = 0; j < len; j++) {
                int b0 = b[boff];
                int b1 = b[boff + 1];
                c[off + j] = (char)((b0 << 8) | b1);
                boff += 2;
            }
        } else {
            for (int j = 0; j < len; j++) {
                int b0 = b[boff];
                int b1 = b[boff + 1];
                c[off + j] = (char)((b1 << 8) | b0);
                boff += 2;
            }
        }
    }

    public void readFully(int[] i, int off, int len) throws IOException {
        if (len < 0 || off + len > i.length) {
            throw new IllegalArgumentException();
        }

        byte[] b = new byte[4*len];
        readFully(b, 0, 4*len);

        int boff = 0;
        if (networkByteOrder) {
            for (int j = 0; j < len; j++) {
                int b0 = (int)b[boff];
                int b1 = (int)b[boff + 1] & 0xff;
                int b2 = (int)b[boff + 2] & 0xff;
                int b3 = (int)b[boff + 3] & 0xff;
                i[off + j] = (b0 << 24) | (b1 << 16) | (b2 << 8) | b3;
                boff += 4;
            }
        } else {
            for (int j = 0; j < len; j++) {
                int b0 = (int)b[boff] & 0xff;
                int b1 = (int)b[boff + 1] & 0xff;
                int b2 = (int)b[boff + 2] & 0xff;
                int b3 = (int)b[boff + 3];
                i[off + j] = (b3 << 24) | (b2 << 16) | (b1 << 8) | b0;
                boff += 4;
            }
        }
    }

    public void readFully(long[] l, int off, int len) throws IOException {
        if (len < 0 || off + len > l.length) {
            throw new IllegalArgumentException();
        }

        byte[] b = new byte[8*len];
        readFully(b, 0, 8*len);

        int boff = 0;
        if (networkByteOrder) {
            for (int j = 0; j < len; j++) {
                int b0 = (int)b[boff];
                int b1 = (int)b[boff + 1] & 0xff;
                int b2 = (int)b[boff + 2] & 0xff;
                int b3 = (int)b[boff + 3] & 0xff;
                int b4 = (int)b[boff + 4];
                int b5 = (int)b[boff + 5] & 0xff;
                int b6 = (int)b[boff + 6] & 0xff;
                int b7 = (int)b[boff + 7] & 0xff;
                
                int i0 = (b0 << 24) | (b1 << 16) | (b2 << 8) | b3;
                int i1 = (b4 << 24) | (b5 << 16) | (b6 << 8) | b7;
                
                l[off + j] = ((long)i0 << 32) | (i1 & 0xffffffffL);
                boff += 8;
            }
        } else {
            for (int j = 0; j < len; j++) {
                int b0 = (int)b[boff] & 0xff;
                int b1 = (int)b[boff + 1] & 0xff;
                int b2 = (int)b[boff + 2] & 0xff;
                int b3 = (int)b[boff + 3];
                int b4 = (int)b[boff + 4] & 0xff;
                int b5 = (int)b[boff + 5] & 0xff;
                int b6 = (int)b[boff + 6] & 0xff;
                int b7 = (int)b[boff + 7];
                
                int i0 = (b7 << 24) | (b6 << 16) | (b5 << 8) | b4;
                int i1 = (b3 << 24) | (b2 << 16) | (b1 << 8) | b0;
                
                l[off + j] = ((long)i0 << 32) | (i1 & 0xffffffffL);
                boff += 8;
            }
        }
    }

    public void readFully(float[] f, int off, int len) throws IOException {
        if (len < 0 || off + len > f.length) {
            throw new IllegalArgumentException();
        }

        byte[] b = new byte[4*len];
        readFully(b, 0, 4*len);

        int boff = 0;
        if (networkByteOrder) {
            for (int j = 0; j < len; j++) {
                int b0 = (int)b[boff];
                int b1 = (int)b[boff + 1] & 0xff;
                int b2 = (int)b[boff + 2] & 0xff;
                int b3 = (int)b[boff + 3] & 0xff;
                int i = (b0 << 24) | (b1 << 16) | (b2 << 8) | b3;
                f[off + j] = Float.intBitsToFloat(i);
                boff += 4;
            }
        } else {
            for (int j = 0; j < len; j++) {
                int b0 = (int)b[boff] & 0xff;
                int b1 = (int)b[boff + 1] & 0xff;
                int b2 = (int)b[boff + 2] & 0xff;
                int b3 = (int)b[boff + 3];
                int i = (b3 << 24) | (b2 << 16) | (b1 << 8) | b0;
                f[off + j] = Float.intBitsToFloat(i);
                boff += 4;
            }
        }
    }

    public void readFully(double[] d, int off, int len) throws IOException {
        if (len < 0 || off + len > d.length) {
            throw new IllegalArgumentException();
        }

        byte[] b = new byte[8*len];
        readFully(b, 0, 8*len);

        int boff = 0;
        if (networkByteOrder) {
            for (int j = 0; j < len; j++) {
                int b0 = (int)b[boff];
                int b1 = (int)b[boff + 1] & 0xff;
                int b2 = (int)b[boff + 2] & 0xff;
                int b3 = (int)b[boff + 3] & 0xff;
                int b4 = (int)b[boff + 4];
                int b5 = (int)b[boff + 5] & 0xff;
                int b6 = (int)b[boff + 6] & 0xff;
                int b7 = (int)b[boff + 7] & 0xff;
                
                int i0 = (b0 << 24) | (b1 << 16) | (b2 << 8) | b3;
                int i1 = (b4 << 24) | (b5 << 16) | (b6 << 8) | b7;
                long l = ((long)i0 << 32) | (i1 & 0xffffffffL);

                d[off + j] = Double.longBitsToDouble(l);
                boff += 8;
            }
        } else {
            for (int j = 0; j < len; j++) {
                int b0 = (int)b[boff] & 0xff;
                int b1 = (int)b[boff + 1] & 0xff;
                int b2 = (int)b[boff + 2] & 0xff;
                int b3 = (int)b[boff + 3];
                int b4 = (int)b[boff + 4] & 0xff;
                int b5 = (int)b[boff + 5] & 0xff;
                int b6 = (int)b[boff + 6] & 0xff;
                int b7 = (int)b[boff + 7];
                
                int i0 = (b7 << 24) | (b6 << 16) | (b5 << 8) | b4;
                int i1 = (b3 << 24) | (b2 << 16) | (b1 << 8) | b0;
                long l = ((long)i0 << 32) | (i1 & 0xffffffffL);

                d[off + j] = Double.longBitsToDouble(l);
                boff += 8;
            }
        }
    }

    public long getStreamPosition() throws IOException {
        checkClosed();
        return streamPos;
    }

    public int getBitOffset() throws IOException {
        checkClosed();
        return bitOffset;
    }

    public void setBitOffset(int bitOffset) throws IOException {
        checkClosed();
        if (bitOffset < 0 || bitOffset > 7) {
            throw new IllegalArgumentException();
        }
        this.bitOffset = bitOffset;
    }

    public int readBit() throws IOException {
        checkClosed();

        long pos = getStreamPosition();
        int val = read();
        if (val == -1) {
            throw new EOFException();
        }

        if (++bitOffset == 8) {
            bitOffset = 0;
        } else {
            seek(pos); // Go back to previous position
        }

        return (val >> (7 - bitOffset)) & 0x1;
    }

    public long readBits(int numBits) throws IOException {
        checkClosed();

        if (numBits < 0 || numBits > 64) {
            throw new IllegalArgumentException();
        }
        if (numBits == 0) {
            return 0L;
        }

        // Have to read additional bits on the left equal to the bit offset
        int bitsToRead = numBits + bitOffset;

        // Read a byte at a time, accumulate
        long accum = 0L;
        while (bitsToRead > 0) {
            int val = read();
            if (val == -1) {
                throw new EOFException();
            }

            accum <<= 8;
            accum |= val;
            bitsToRead -= 8;
        }

        // Update bit offset
        this.bitOffset = (this.bitOffset + numBits) & 0x7;

        // Move byte position back if in the middle of a byte
        if (bitOffset != 0) {
            seek(getStreamPosition() - 1);
        }

        // Shift away unwanted bits on the right.
        accum >>>= (-bitsToRead); // Negative of bitsToRead == extra bits read
        
        // Mask out unwanted bits on the left
        accum &= (-1L >>> (64 - numBits));

        return accum;
    }

    /**
     * Returns <code>-1L</code> to indicate that the stream has unknown
     * length.  Subclasses must override this method to provide actual
     * length information.
     *
     * @return -1L to indicate unknown length.
     */
    public long length() {
        return -1L;
    }

    /**
     * Advances the current stream position by calling
     * <code>seek(getStreamPosition() + n)</code>.
     *
     * <p> The bit offset is reset to zero.
     *
     * @param n the number of bytes to seek forward.
     *
     * @return an <code>int</code> representing the number of bytes
     * skipped.
     *
     * @exception IOException if <code>getStreamPosition</code>
     * throws an <code>IOException</code> when computing either
     * the starting or ending position.
     */
    public int skipBytes(int n) throws IOException {
        long pos = getStreamPosition();
        seek(pos + n);
        return (int)(getStreamPosition() - pos);
    }

    /**
     * Advances the current stream position by calling
     * <code>seek(getStreamPosition() + n)</code>.
     *
     * <p> The bit offset is reset to zero.
     *
     * @param n the number of bytes to seek forward.
     *
     * @return a <code>long</code> representing the number of bytes
     * skipped.
     *
     * @exception IOException if <code>getStreamPosition</code>
     * throws an <code>IOException</code> when computing either
     * the starting or ending position.
     */
    public long skipBytes(long n) throws IOException {
        long pos = getStreamPosition();
        seek(pos + n);
        return getStreamPosition() - pos;
    }

    /**
     * Sets the current stream position and resets the bit offset to
     * 0.
     *
     * @exception IndexOutOfBoundsException if <code>pos</code> is smaller
     * the the flushed position.
     * @exception EOFException if the end of the stream is passed.
     * @exception IOException if any other I/O error occurs.
     */
    public void seek(long pos) throws IOException {
        checkClosed();

        // This test also covers pos < 0
        if (pos < flushedPos) {
            throw new IndexOutOfBoundsException();
        }

        this.streamPos = pos;
        this.bitOffset = 0;
    }

    /**
     * Pushes the current stream position onto a stack of marked
     * positions.
     */
    public void mark() {
        try {
            markByteStack.push(new Long(getStreamPosition()));
            markBitStack.push(new Integer(getBitOffset()));
        } catch (IOException e) {
        }
    }

    /**
     * Resets the current stream byte and bit positions from the stack
     * of marked positions.
     */
    public void reset() throws IOException {
        if (markByteStack.empty()) {
            return;
        }

        Long pos = (Long)markByteStack.pop();
        seek(pos.longValue());

        Integer offset = (Integer)markBitStack.pop();
        setBitOffset(offset.intValue());
    }

    public void flushBefore(long pos) throws IOException {
        // Invariant: flushedPos >= 0
        flushedPos = Math.max(flushedPos, pos);
    }

    public void flush() throws IOException {
        flushBefore(getStreamPosition());
    }

    public long getFlushedPosition() {
        return flushedPos;
    }

    /** 
     * Default implementation returns false.  Subclasses should
     * override this if they cache data.
     */
    public boolean isCached() {
        return false;
    }

    /** 
     * Default implementation returns false.  Subclasses should
     * override this if they cache data in main memory.
     */
    public boolean isCachedMemory() {
        return false;
    }

    /** 
     * Default implementation returns false.  Subclasses should
     * override this if they cache data in a temporary file.
     */
    public boolean isCachedFile() {
        return false;
    }

    public void close() throws IOException {
        checkClosed();

        isClosed = true;
    }

    protected void finalize() {
        if (!isClosed) {
            try {
                close();
            } catch (IOException e) {
            }
        }
    }
}
