/*
 * @(#)MemoryCache.java	1.4 00/10/25
 *
 * Copyright 2000 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

package javax.imageio.stream;

import java.util.ArrayList;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;

/**
 * Package-visible class consolidating common code for 
 * <code>MemoryCacheImageInputStream</code> and
 * <code>MemoryCacheImageOutputStream</code>.
 * This class keeps an <code>ArrayList</code> of 8K blocks,
 * loaded sequentially.  Blocks may only be disposed of
 * from the index 0 forward.  As blocks are freed, the
 * corresponding entries in the array list are set to 
 * <code>null</code>, but no compacting is performed.
 * This allows the index for each block to never change,
 * and the length of the cache is always the same as the
 * total amount of data ever cached.  Cached data is
 * therefore always contiguous from the point of last
 * disposal to the current length.
 *
 * A <code>MemoryCache</code> may be reused after a call
 * to <code>reset()</code>.
 *
 * @version 0.5
 */
class MemoryCache {

    private static final int BUFFER_LENGTH = 8192;

    private ArrayList cache = new ArrayList();

    /**
     * The largest position ever written to the cache.
     */
    private long length = 0L;

    /**
     * Ensures that at least <code>pos</code> bytes are cached,
     * or the end of the source is reached.  The return value
     * is equal to the smaller of <code>pos</code> and the
     * length of the source.
     */
    public long loadFromStream(InputStream stream, long pos) 
        throws IOException {

        // We've already got enough data cached
        if (pos < length) {
            return pos;
        }

        int offset = (int) length%BUFFER_LENGTH;
        byte [] buf = null;

        long len = pos - length;

        if (offset != 0) {
            buf = (byte []) cache.get((int)(length/BUFFER_LENGTH));
        }

        while (len > 0) {
            if (buf == null) {
                buf = new byte[BUFFER_LENGTH];
                offset = 0;
            }
            
            int left = BUFFER_LENGTH - offset;
            int nbytes = 
                stream.read(buf, offset, (int)Math.min(len, (long)left));
            if (nbytes == -1) {
                return length; // EOF
            }

            if (offset == 0) {
                cache.add(buf);
            }
            buf = null;

            len -= nbytes;
            length += nbytes;
        }

        return pos;
    }

    /**
     * Writes out a portion of the cache to an <code>OutputStream</code>.
     * This method preserves no state about the output stream, and does
     * not dispose of any blocks containing bytes written.  To dispose
     * blocks, use {@link #disposeBefore <code>disposeBefore()</code>}.
     *
     * @exception IndexOutOfBoundsException if any portion of
     * the requested data is not in the cache (including if <code>pos</code>
     * is in a block already disposed), or if either <code>pos</code> or 
     * <code>len</code> is < 0.
     */
    public void writeToStream(OutputStream stream, long pos, long len) 
        throws IOException {
        if ((pos >= length) || (pos+len>=length)) {
            throw new IndexOutOfBoundsException("Argument out of cache");
        }
        if ((pos < 0) || (len < 0)) {
            throw new IndexOutOfBoundsException("negative pos or len");
        }

        int bufIndex = (int)(pos/BUFFER_LENGTH);
        int offset = (int)(pos%BUFFER_LENGTH);
        
        byte [] buf = (byte[]) cache.get(bufIndex++);

        if (buf == null) {
            throw new IndexOutOfBoundsException("pos already disposed");
        }

        while (len > 0) {
            if (buf == null) {
                buf = (byte[]) cache.get(bufIndex++);
                offset = 0;
            }
            int nbytes = (int)Math.min(len, (long)(BUFFER_LENGTH - offset));
            stream.write(buf, offset, nbytes);
            buf = null;
            len -= nbytes;
        }
    }

    /**
     * Load the cache from a byte array by copying the data, 
     * appending it to the current cache contents.  Returns
     * the new length of the cache.
     *
     * @exception NullPointerException if <code>b</code> is <code>null</code>
     * @exception IndexOutOfBoundsException if <code>off</code>
     * or <code>len</code> are < 0
     * or if <code>off+len > b.length</code>.
     */
    public long loadFromArray(byte [] b, int off, int len) {
        if (b == null) {
            throw new NullPointerException("b == null!");
        }
        if ((off < 0) || (len < 0) || (off + len > b.length)) {
            throw new IndexOutOfBoundsException();
        }

        if (len == 0) {
            return length;
        }

        int offset = (int) (length%BUFFER_LENGTH);
        byte [] buf = null;

        if (offset != 0) {
            buf = (byte[]) cache.get((int)(length/BUFFER_LENGTH));
        }

        while (len > 0) {
            if (buf == null) {
                buf = new byte[BUFFER_LENGTH];
                offset = 0;
            }
            int nbytes = Math.min(len, BUFFER_LENGTH-offset);
            System.arraycopy(b, off, buf, offset, nbytes);
            if (offset == 0) {
                cache.add(buf);
            }
            buf = null;
            off+=nbytes;
            len -= nbytes;
            length += nbytes;
        }
        return length;
    }

    /**
     * Append a single byte to the cache from the lower 8 bits of
     * the argument.  The upper 24 bits are ignored.
     */
    public void load(int b) {
        int offset = (int) (length%BUFFER_LENGTH);
        byte [] buf = null;
        if (offset == 0) {
            buf = new byte[BUFFER_LENGTH];
            cache.add(buf);
        } else {
            buf = (byte[]) cache.get((int)(length/BUFFER_LENGTH));
        }
        buf[offset] = (byte)(b & 0xff);
        length++;
    }

    /**
     * Returns the total length of data that has been cached, 
     * regardless of whether any early blocks have been disposed.
     * This value will only ever increase.
     */
    public long getLength() {
        return length;
    }

    /**
     * Returns the single byte at the given position, as an 
     * <code>int</code>.  Returns -1 if this position has
     * not been cached or has been disposed.
     */
    public int read(long pos) {
        if (pos >= length) {
            return -1;
        }
                               
        byte [] buf = (byte []) cache.get((int)(pos/BUFFER_LENGTH));
        if (buf == null) {
            return -1;
        }

        return ((int)buf[(int)(pos%BUFFER_LENGTH)]) & 0xff;
    }

    /**
     * Copy <code>len</code> bytes from the cache, starting
     * at cache position <code>pos</code>, into the array 
     * <code>b</code> at offset <code>off</code>.
     *
     * @exception NullPointerException if b is <code>null</code>
     * @exception IndexOutOfBoundsException if <code>off</code>
     * or <code>len</code> are < 0
     * or if <code>off+len > b.length</code> or if any portion
     * of the requested data is not in the cache (including
     * if <code>pos</code> is in a block that has already
     * been disposed).
     */
    public void read(byte [] b, int off, int len, long pos) {
        if (b == null) {
            throw new NullPointerException("b == null!");
        }
        if ((off < 0) || (len < 0) || (off + len > b.length)) {
            throw new IndexOutOfBoundsException();
        }
        if (pos+len > length) {
            throw new IndexOutOfBoundsException();
        }
        
        int index = (int)(pos/BUFFER_LENGTH);
        int offset = (int)pos%BUFFER_LENGTH;

        while (len > 0) {

            int pieceLen = Math.min(len, BUFFER_LENGTH - offset);
            System.arraycopy((byte [])cache.get(index++),
                             offset, b, off, pieceLen);

            len -= pieceLen;
            off += pieceLen;
            offset = 0;  // Always after the first
        }
    }

    /**
     * Free the blocks up to the position <code>pos</code>.
     * The byte at <code>pos</code> remains available.
     * 
     * @exception IndexOutOfBoundsException if <code>pos</code>
     * is in a block that has already been disposed.
     */
    public void disposeBefore(long pos) {
        int index = (int)(pos/BUFFER_LENGTH);
        if (cache.get(index) == null) {
            throw new IndexOutOfBoundsException("pos already disposed");
        }
        for (int i = 0; i < index; i++) {
            cache.set(i, null);
        }
    }

    /**
     * Erase the entire cache contents and reset the length to 0.
     * The cache object may subsequently be reused as though it had just
     * been allocated.
     */
    public void reset() {
        cache.clear();
        length = 0L;
    }
 }
