/*
 * @(#)MemoryCacheImageOutputStream.java	1.5 00/10/20
 *
 * Copyright 2000 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

package javax.imageio.stream;

import java.io.IOException;
import java.io.OutputStream;

/**
 * An implementation of <code>ImageOutputStream</code> that writes its
 * output to a regular <code>OutputStream</code>.  A memory buffer is
 * used to cache at least the data between the discard position and
 * the current write position.  The only constructor takes an 
 * <code>OutputStream</code>, so this class may not be used for 
 * read/modify/write operations.  Reading can occur only on parts of 
 * the stream that have already been written to the cache and not
 * yet flushed.
 *
 * @version 0.5
 */
public class MemoryCacheImageOutputStream extends ImageOutputStreamImpl {

    private OutputStream stream;

    private MemoryCache cache = new MemoryCache();

    /**
     * Constructs a <code>MemoryCacheImageOutputStream</code> that will write
     * to a given <code>OutputStream</code>.
     *
     * @param stream an <code>OutputStream</code> to write to.
     *
     * @exception IllegalArgumentException if <code>stream</code> is
     * <code>null</code>.
     */
    public MemoryCacheImageOutputStream(OutputStream stream) {
        if (stream == null) {
            throw new IllegalArgumentException("stream == null!");
        }
        this.stream = stream;
    }

    public int read() throws IOException {
        checkClosed();

        int val = cache.read(streamPos);
        if (val != -1) {
            ++streamPos;
            bitOffset = 0;
        }
        return val;
    }

    public int read(byte[] b, int off, int len) throws IOException {
        checkClosed();

        len = (int)Math.min (cache.getLength() - streamPos, (long)len);

        cache.read(b, off, len, streamPos);
        
        streamPos += len;
        bitOffset = 0;

        return len;
    }

    public void write(int b) throws IOException {
        checkClosed();
        flushBits();
        cache.load(b);
        ++streamPos;
    }

    public void write(byte[] b, int off, int len) throws IOException {
        checkClosed();
        flushBits();
        cache.loadFromArray(b, off, len);
        streamPos += len;
    }

    public long length() {
        return cache.getLength();
    }

    public boolean isCached() {
        return true;
    }

    public boolean isCachedMemory() {
        return true;
    }

    /**
     * Closes this <code>MemoryCacheImageOututStream</code>.  All
     * pending data is flushed to the output, adn the cache
     * is released.  The destination <code>OutputStream</code>
     * is not closed.
     */
    public void close() throws IOException {
        long length = cache.getLength();
        flushBefore(length);
        super.close();
        cache.reset();
        stream = null;
    }

    public void flushBefore(long pos) throws IOException {
        long maxPos = cache.getLength();
        if (pos > maxPos) {
            // Shouldn't this throw an out of bounds exception?
            pos = maxPos;
        }
        
        long len = pos - flushedPos;

        cache.writeToStream(stream, flushedPos, len);

        if (streamPos < pos) {
            streamPos = pos;
        }

        super.flushBefore(pos);
    }
}
