/*
 * @(#)GIFAnimator.java	1.3 00/11/16
 *
 * Copyright 2000 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.Iterator;
import javax.imageio.IIOException;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.stream.ImageInputStream;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.Timer;

// Use the GIF plug-ins private APIs
import com.sun.imageio.plugins.gif.GIFImageMetadata;
import com.sun.imageio.plugins.gif.GIFStreamMetadata;

class GIFAnimatorWindowAdapter extends WindowAdapter {

    public GIFAnimatorWindowAdapter() {}
    
    public void windowClosing(WindowEvent e) {
        System.exit(0);
    }
}

/**
 * A JPanel that displays a GIF animation.  Clicking the mouse on
 * the panel will toggle the animation.
 *
 * <p> The current implementation accesses the internal fields of the
 * GIF plug-in's metadata object directly.  It is also possible to get
 * the required information using the IIOMetadata.getAsTree method.
 *
 * @version 0.5
 *
 * @author Daniel Rice
 */
public class GIFAnimator extends JPanel
    implements ActionListener, MouseListener {

    // The current frame number
    int frame;

    // If true, advance the frame on each timer event
    boolean animating = true;

    // The input source
    ImageInputStream iis;

    // The ImageReader
    ImageReader reader = null;

    // The overall dimensions of the animation
    int screenWidth;
    int screenHeight;

    // The position, disposal method, and delay time of the current frame
    int leftPosition;
    int topPosition;
    int disposalMethod;
    int delayTime;
    
    // The backgound color
    Color backgroundColor = Color.black;

    // The background images and a Graphics used to draw into it
    BufferedImage backgroundImage = null;
    Graphics backgroundGraphics;

    // The current frame image
    BufferedImage frameImage = null;

    // A Swing timer to control the animation
    Timer timer;

    public GIFAnimator(File f) throws IIOException {
        // Get an ImageInputStream
        this.iis = null;
        try {
            iis = ImageIO.createImageInputStream(f);
        } catch (IOException ioe) {
            ioe.printStackTrace();
        }
        if (iis == null) {
            throw new IIOException("Unable to get a stream!");
        }
        
        // Get an ImageReader
        Iterator iter = ImageIO.getImageReaders(iis);
        while (iter.hasNext()) {
            this.reader = (ImageReader)iter.next();
            break;
        }
        if (reader == null) {
            throw new IIOException("Unable to find a reader!");
        }

        // Get a Swing timer to control the animation
        this.timer = new Timer(1, this);
        timer.setInitialDelay(0);
        timer.setCoalesce(true);
        timer.start();

        // Listen for mouse clicks
        addMouseListener(this);

        // Display the filename as a tool tip
        setToolTipText(f.toString());

        // Seek to the first frame
        restart();
    }

    private synchronized void restart() throws IIOException {
        // Seek to beginning of stream
        try {
            iis.seek(0L);
        } catch (Exception e) {
            e.printStackTrace();
        }

        // Reset the reader
        reader.reset();
        reader.setInput(iis, true);

        // Retrieve the overall image dimensions from the stream metadata
	IIOMetadata metadata = reader.getStreamMetadata();
	if (metadata == null || !(metadata instanceof GIFStreamMetadata)) {
	    System.out.println("Not a GIF file!");
	    System.exit(0);
        }
        GIFStreamMetadata streamMetadata = (GIFStreamMetadata)metadata;
        this.screenWidth = streamMetadata.logicalScreenWidth;
        this.screenHeight = streamMetadata.logicalScreenHeight;

        // Retrieve the background color
        int backgroundIndex = streamMetadata.backgroundColorIndex;
        byte[] colorTable = streamMetadata.globalColorTable;
        if (colorTable != null) {
            if (3*backgroundIndex + 2 >= colorTable.length) {
                backgroundIndex = colorTable.length/3 - 1;
            }
            
            int backgroundRed = colorTable[3*backgroundIndex] & 0xff;
            int backgroundGreen = colorTable[3*backgroundIndex + 1] & 0xff;
            int backgroundBlue = colorTable[3*backgroundIndex + 2] & 0xff;

            this.backgroundColor = new Color(backgroundRed,
                                             backgroundGreen,
                                             backgroundBlue);
        }

        // Create a persistent image buffer and Graphics object
        this.backgroundImage = new BufferedImage(screenWidth, screenHeight,
                                                 BufferedImage.TYPE_INT_BGR);
        this.backgroundGraphics = backgroundImage.createGraphics();
        backgroundGraphics.setColor(backgroundColor);
        backgroundGraphics.fillRect(0, 0, screenWidth, screenHeight);
    }

    public Dimension getPreferredSize() {
        return new Dimension(screenWidth, screenHeight);
    }

    private synchronized void advanceFrame() {
        try {
            if (frame == 0) {
                restart();
            }
            
            // Get the frame position and diposal method from the
            // image metadata
            GIFImageMetadata metadata =
                (GIFImageMetadata)reader.getImageMetadata(frame);
            this.leftPosition = metadata.imageLeftPosition;
            this.topPosition = metadata.imageTopPosition;
            this.disposalMethod = metadata.disposalMethod;
            // Treat method 0 (unspecified) as if it were 1
            if (disposalMethod == 0) {
                disposalMethod = 1;
            }
            
            // Read the frame contents
            frameImage = reader.read(frame, null);
            int w = frameImage.getWidth();
            int h = frameImage.getHeight();
            
            if (disposalMethod == 1) {
                // Incorporate current frame into the master image
                backgroundGraphics.drawImage(frameImage,
                                             leftPosition, topPosition, null);
            } else if (disposalMethod == 2) {
                // Leave a blank space in the master image
                backgroundGraphics.fillRect(leftPosition, topPosition, w, h);
            } // if disposalMethod == 3, do nothing

            // Advance the frame counter
            ++frame;

            // Set timer to next interframe delay
            metadata =
                (GIFImageMetadata)reader.getImageMetadata(frame);
            this.delayTime = metadata.delayTime;
            timer.setDelay(10*delayTime);
        } catch (IIOException e) {
            // Something bad happened
            e.printStackTrace();
        } catch (IndexOutOfBoundsException e) {
            // We've attempted to go past the last frame, restart
            timer.setDelay(0);
            frame = 0;
        }
    }

    public synchronized void paint(Graphics g) {
        if (backgroundImage != null && frameImage != null) {
            // Draw the background image
            g.drawImage(backgroundImage, 0, 0, null);
            // Draw the current frame on top
            g.drawImage(frameImage, leftPosition, topPosition, null);
        }
    }

    /**
     * Start the animation timer.
     */
    public void start() {
        timer.start();
    }

    /**
     * Stop the animation timer.
     */
    public void stop() {
        timer.stop();
    }
    
    /**
     * Control whether the image is being animated.
     */
    public void setAnimating(boolean animating) {
        this.animating = animating;
    }

    /**
     * Returns whether the image is being animated.
     */
    public boolean getAnimating() {
        return animating;
    }

    // Called by the Swing timer
    public void actionPerformed(ActionEvent e) {
        if (animating) {
            advanceFrame();
            repaint();
        }
    }

    // On a mouse click, toggle animation
    public synchronized void mouseClicked(MouseEvent e) {
        animating = !animating;
    }

    public void mouseEntered(MouseEvent e) {
    }
    
    public void mouseExited(MouseEvent e) {
    }

    public void mousePressed(MouseEvent e) {
    }
    
    public void mouseReleased(MouseEvent e) {
    }
  
    public static void main(String[] args) {
        if (args.length != 1) {
            System.err.println("usage: java GIFAnimator filename");
            System.exit(0);
        }
        
        File f = new File(args[0]);
        if (!f.exists()) {
            System.out.println("File " + f + " does not exist!");
            System.exit(0);
        }

        try {
            GIFAnimator gifanim = new GIFAnimator(f);
            
            JFrame jf = new JFrame();
            jf.addWindowListener(new GIFAnimatorWindowAdapter());
            jf.getContentPane().add(gifanim);
            jf.setLocation(100, 100);
            jf.pack();
            jf.setVisible(true);

            gifanim.start();
        } catch (IIOException iio) {
            System.err.println("IIOException: " + iio);
            System.exit(0);
        }

        // Sleep main thread until animation window is destroyed
        while (true) {
            try {
                Thread.currentThread().sleep(1000);
            } catch (InterruptedException e) {
            }
        }
    }
}
