/*
 * @(#)IIOMetadataFormatImpl.java	1.9 00/10/19
 *
 * Copyright 2000 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

package javax.imageio.metadata;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.imageio.ImageTypeSpecifier;

/**
 * A concrete class providing a reusable implementation of the
 * <code>IIOMetadataFormat</code> interface.  In addition, a static
 * instance representing the standard, plug-in neutral
 * <code>com.sun.imageio_1.0</code> format is provided by the
 * <code>getStandardFormatInstance</code> method.
 *
 * @version 0.5
 */
public class IIOMetadataFormatImpl implements IIOMetadataFormat {

    private static IIOMetadataFormat standardFormat = null;

    private String rootName;

    // Element name (String) -> Element
    private HashMap elementMap = new HashMap();

    class Element {
        String elementName;

        String parentName;
        int childPolicy;
        int minChildren = 0;
        int maxChildren = 0;

        // Child names (Strings)
        List childList = new ArrayList();

        // List of attribute names in the order they were added
        List attrList = new ArrayList();
        // Attr name (String) -> Attribute
        Map attrMap = new HashMap();

        ObjectValue objectValue;
    }
    
    class Attribute {
        String attrName;

        int valueType = VALUE_ARBITRARY;
        int dataType;
        boolean required;
        String defaultValue = null;

        // enumeration
        List enumeratedValues;

        // range
        String minValue;
        String maxValue;

        // list
        int listMinLength;
        int listMaxLength;
    }

    class ObjectValue {
        int valueType = VALUE_NONE;
        Class classType = null;
        Object defaultValue = null;

        // Meaningful only if valueType == VALUE_ENUMERATION
        List enumeratedValues = null;
        
        // Meaningful only if valueType == VALUE_RANGE
        Comparable minValue = null;
        Comparable maxValue = null;

        // Meaningful only if valueType == VALUE_LIST
        int arrayMinLength = 0;
        int arrayMaxLength = 0;
    }

    /**
     * Constructs a blank <code>IIOMetadataFormatImpl</code> instance,
     * with a given root element name and child policy (other than
     * <code>CHILD_POLICY_REPEAT</code>).  Additional elements, and
     * their attributes and <code>Object</code> reference information
     * may be added using the various <code>add</code> methods.
     *
     * @param rootName the name of the root element.
     * @param childPolicy one of the <code>CHILD_POLICY_*</code> constants,
     * other than <code>CHILD_POLICY_REPEAT</code>.
     *
     * @exception IllegalArgumentException if <code>rootName</code> is
     * <code>null</code>.
     * @exception IllegalArgumentException if <code>childPolicy</code> is
     * not one of the predefined constants.
     */
    public IIOMetadataFormatImpl(String rootName,
                                 int childPolicy) {
        if (rootName == null) {
            throw new IllegalArgumentException("rootName == null!");
        }
        if (childPolicy < CHILD_POLICY_EMPTY ||
            childPolicy >= CHILD_POLICY_REPEAT) {
            throw new IllegalArgumentException("Invalid value for childPolicy!");
        }

        this.rootName = rootName;

        Element root = new Element();
        root.elementName = rootName;
        root.parentName = null;
        root.childPolicy = childPolicy;

        elementMap.put(rootName, root);
    }

    /**
     * Constructs a blank <code>IIOMetadataFormatImpl</code> instance,
     * with a given root element name and a child policy of
     * <code>CHILD_POLICY_REPEAT</code>.  Additional elements, and
     * their attributes and <code>Object</code> reference information
     * may be added using the various <code>add</code> methods.
     *
     * @param rootName the name of the root element.
     * @param minChildren the minimum number of children of the node.
     * @param maxChildren the maximum number of children of the node.
     *
     * @exception IllegalArgumentException if <code>rootName</code> is
     * <code>null</code>.
     * @exception IllegalArgumentException if <code>minChildren</code>
     * is negative or larger than <code>maxChildren</code>.
     */
    public IIOMetadataFormatImpl(String rootName,
                                 int minChildren,
                                 int maxChildren) {
        if (rootName == null) {
            throw new IllegalArgumentException("rootName == null!");
        }
        if (minChildren < 0) {
            throw new IllegalArgumentException("minChildren < 0!");
        }
        if (minChildren > maxChildren) {
            throw new IllegalArgumentException("minChildren > maxChildren!");
        }

        Element root = new Element();
        root.elementName = rootName;
        root.parentName = null;
        root.childPolicy = CHILD_POLICY_REPEAT;
        root.minChildren = minChildren;
        root.maxChildren = maxChildren;

        this.rootName = rootName;
        elementMap.put(rootName, root);
    }

    // Standard format descriptor

    // Utility method for nodes with a single atttribute named "value"
    private static void addSingleAttributeElement(IIOMetadataFormatImpl impl,
                                                  String elementName,
                                                  String parentName,
                                                  int dataType) {
        impl.addElement(elementName, parentName, CHILD_POLICY_EMPTY);
        impl.addAttribute(elementName, "value", dataType, true, null);
    }

    private synchronized static void createStandardFormat() {
        if (standardFormat == null) {
            List values;

            IIOMetadataFormatImpl impl =
                new IIOMetadataFormatImpl("com.sun.imageio_1.0",
                                          CHILD_POLICY_SOME);
            
            // root -> Chroma
            impl.addElement("Chroma", "com.sun.imageio_1.0",
                         CHILD_POLICY_SOME);

            // root -> Chroma -> ColorSpaceType
            impl.addElement("ColorSpaceType", "Chroma",
                         CHILD_POLICY_EMPTY);

            values = new ArrayList();
            values.add("XYZ");
            values.add("Lab");
            values.add("Luv");
            values.add("YCbCr");
            values.add("Yxy");
            values.add("RGB");
            values.add("GRAY");
            values.add("HSV");
            values.add("HLS");
            values.add("CMYK");
            values.add("CMY");
            values.add("2CLR");
            values.add("3CLR");
            values.add("4CLR");
            values.add("5CLR");
            values.add("6CLR");
            values.add("7CLR");
            values.add("8CLR");
            values.add("9CLR");
            values.add("ACLR");
            values.add("BCLR");
            values.add("CCLR");
            values.add("DCLR");
            values.add("ECLR");
            values.add("FCLR");
            impl.addAttribute("ColorSpaceType",
                              "name",
                              DATATYPE_STRING,
                              true,
                              null,
                              values);

            // root -> Chroma -> BlackIsZero
            impl.addElement("BlackIsZero", "Chroma", CHILD_POLICY_EMPTY);
            impl.addBooleanAttribute("BlackIsZero", "value", true, true);

            // root -> Chroma -> Palette
            impl.addElement("Palette", "Chroma", 0, Integer.MAX_VALUE);

            // root -> Chroma -> PaletteEntry
            impl.addElement("PaletteEntry", "Palette", CHILD_POLICY_EMPTY);
            impl.addAttribute("PaletteEntry", "index", DATATYPE_INTEGER,
                              true, null);
            impl.addAttribute("PaletteEntry", "red", DATATYPE_INTEGER,
                              true, null);
            impl.addAttribute("PaletteEntry", "green", DATATYPE_INTEGER,
                              true, null);
            impl.addAttribute("PaletteEntry", "blue", DATATYPE_INTEGER,
                              true, null);

            // root -> Chroma -> BackgroundIndex
            impl.addElement("BackgroundIndex", "Chroma", CHILD_POLICY_EMPTY);
            impl.addAttribute("BackgroundIndex", "value", DATATYPE_INTEGER,
                              true, 0, Integer.MAX_VALUE);

            // root -> Chroma -> BackgroundColor
            impl.addElement("BackgroundColor", "Chroma", CHILD_POLICY_EMPTY);
            impl.addAttribute("BackgroundColor", "value", DATATYPE_INTEGER,
                              true, 0, Integer.MAX_VALUE);

            // root -> Compression
            impl.addElement("Compression", "com.sun.imageio_1.0",
                         CHILD_POLICY_SOME);

            // root -> Compression -> CompressionTypeName
            addSingleAttributeElement(impl,
                                      "CompressionTypeName",
                                      "Compression",
                                      DATATYPE_STRING);
            // root -> Compression -> Lossless
            impl.addElement("Lossless", "Compression", CHILD_POLICY_EMPTY);
            impl.addBooleanAttribute("Lossless", "value", true, true);

            // root -> Compression -> NumProgressiveScans
            addSingleAttributeElement(impl,
                                      "NumProgressiveScans",
                                      "Compression",
                                      DATATYPE_INTEGER);

            // root -> Compression -> BitRate
            addSingleAttributeElement(impl,
                                      "BitRate",
                                      "Compression",
                                      DATATYPE_FLOAT);
            // root -> Data
            impl.addElement("Data", "com.sun.imageio_1.0",
                         CHILD_POLICY_SOME);
            
            // root -> Data -> PlanarConfiguration
            impl.addElement("PlanarConfiguration", "Data", CHILD_POLICY_EMPTY);

            values = new ArrayList();
            values.add("PixelInterleaved");
            values.add("PlaneInterleaved");
            values.add("LineInterleaved");
            values.add("TileInterleaved");
            impl.addAttribute("PlanarConfiguration", "value",
                              DATATYPE_STRING,
                              true,
                              null,
                              values);

            // root -> Data -> SampleFormat
            impl.addElement("SampleFormat", "Data", CHILD_POLICY_EMPTY);

            values = new ArrayList();
            values.add("SignedIntegral");
            values.add("UnsignedIntegral");
            values.add("Real");
            values.add("Index");
            impl.addAttribute("SampleFormat", "value",
                              DATATYPE_STRING,
                              true,
                              null,
                              values);

            // root -> Data -> BitsPerSample
            addSingleAttributeElement(impl,
                                      "BitsPerSample",
                                      "Data",
                                      DATATYPE_INTEGER);

            // root -> Data -> SignificantBitsPerSample
            addSingleAttributeElement(impl,
                                      "SignificantBitsPerSample",
                                      "Data",
                                      DATATYPE_INTEGER);

            // root -> Data -> SampleMSB
            addSingleAttributeElement(impl,
                                      "SampleMSB",
                                      "Data",
                                      DATATYPE_INTEGER);

            // root -> Dimension
            impl.addElement("Dimension", "com.sun.imageio_1.0",
                         CHILD_POLICY_SOME);

            // root -> Dimension -> PixelAspectRatio
            addSingleAttributeElement(impl,
                                      "PixelAspectRatio",
                                      "Dimension",
                                      DATATYPE_FLOAT);

            // root -> Dimension -> ImageOrientation
            impl.addElement("ImageOrientation", "Dimension",
                            CHILD_POLICY_EMPTY);

            values = new ArrayList();
            values.add("normal");
            values.add("flip_y");
            values.add("flip_x_y");
            values.add("flip_x");
            values.add("flip_x_rot90");
            values.add("rot_270");
            values.add("rot90_flip_x");
            values.add("rot90");
            impl.addAttribute("ImageOrientation", "value",
                              DATATYPE_STRING,
                              true,
                              null,
                              values);

            // root -> Dimension -> HorizontalPixelSize
            addSingleAttributeElement(impl,
                                      "HorizontalPixelSize",
                                      "Dimension",
                                      DATATYPE_FLOAT);

            // root -> Dimension -> VerticalPixelSize
            addSingleAttributeElement(impl,
                                      "VerticalPixelSize",
                                      "Dimension",
                                      DATATYPE_FLOAT);

            // root -> Dimension -> HorizontalPhysicalPixelSpacing
            addSingleAttributeElement(impl,
                                      "HorizontalPhysicalPixelSpacing",
                                      "Dimension",
                                      DATATYPE_FLOAT);

            // root -> Dimension -> VerticalPhysicalPixelSpacing
            addSingleAttributeElement(impl,
                                      "VerticalPhysicalPixelSpacing",
                                      "Dimension",
                                      DATATYPE_FLOAT);

            // root -> Dimension -> HorizontalPosition
            addSingleAttributeElement(impl,
                                      "HorizontalPosition",
                                      "Dimension",
                                      DATATYPE_FLOAT);

            // root -> Dimension -> VerticalPosition
            addSingleAttributeElement(impl,
                                      "VerticalPosition",
                                      "Dimension",
                                      DATATYPE_FLOAT);

            // root -> Dimension -> HorizontalPixelOffset
            addSingleAttributeElement(impl,
                                      "HorizontalPixelOffset",
                                      "Dimension",
                                      DATATYPE_FLOAT);

            // root -> Dimension -> VerticalPixelOffset
            addSingleAttributeElement(impl,
                                      "VerticalPixelOffset",
                                      "Dimension",
                                      DATATYPE_FLOAT);


            // root -> Document
            impl.addElement("Document", "com.sun.imageio_1.0",
                         CHILD_POLICY_SOME);

            // root -> Document -> SubimageInterpretation
            impl.addElement("SubimageInterpretation", "Document",
                            CHILD_POLICY_EMPTY);
            values = new ArrayList();
            values.add("Standalone");
            values.add("SignificantBitsPerSample");
            values.add("FullResolution");
            values.add("ReducedResolution");
            values.add("PyramidLayer");
            values.add("Preview");
            values.add("VolumeSlice");
            values.add("ObjectView");
            values.add("Panorama");
            values.add("AnimationFrame");
            values.add("TransparencyMask");
            values.add("CompositingLayer");
            values.add("SpectralSlice");
            values.add("Unknown");
            impl.addAttribute("SubimageInterpretation", "value",
                              DATATYPE_STRING,
                              true,
                              null,
                              values);

            // root -> Document -> ImageCreationTime
            addSingleAttributeElement(impl,
                                      "ImageCreationTime",
                                      "Document",
                                      DATATYPE_TIME);

            // root -> Document -> ImageModificationTime
            addSingleAttributeElement(impl,
                                      "ImageModificationTime",
                                      "Document",
                                      DATATYPE_TIME);

            // root -> Text
            impl.addElement("Text", "com.sun.imageio_1.0",
                            0, Integer.MAX_VALUE);

            // root -> Text -> TextEntry
            impl.addElement("TextEntry", "Text", CHILD_POLICY_EMPTY);
            impl.addAttribute("TextEntry", "keyword",
                              DATATYPE_STRING,
                              true,
                              null);
            impl.addAttribute("TextEntry", "value",
                              DATATYPE_STRING,
                              true,
                              null);
            impl.addAttribute("TextEntry", "language",
                              DATATYPE_STRING,
                              false,
                              null);
            impl.addAttribute("TextEntry", "encoding",
                              DATATYPE_STRING,
                              false,
                              null);

            values = new ArrayList();
            values.add("none");
            values.add("lzw");
            values.add("zip");
            values.add("bzip");
            values.add("other");
            impl.addAttribute("TextEntry", "compression",
                              DATATYPE_STRING,
                              false,
                              null,
                              values);

            // root -> Transparency
            impl.addElement("Transparency", "com.sun.imageio_1.0",
                         CHILD_POLICY_SOME);

            // root -> Transparency -> Alpha
            impl.addElement("Alpha", "Transparency", CHILD_POLICY_EMPTY);

            values = new ArrayList();
            values.add("none");
            values.add("premultiplied");
            values.add("nonpremultiplied");
            impl.addAttribute("Alpha", "value",
                              DATATYPE_STRING,
                              false,
                              "none",
                              values);

            // root -> Transparency -> TransparentIndex
            addSingleAttributeElement(impl, "TransparentIndex", "Transparency",
                                      DATATYPE_INTEGER);

            // root -> Transparency -> TransparentColor
            impl.addElement("TransparentColor", "Transparency",
                            CHILD_POLICY_EMPTY);
            impl.addAttribute("TransparentColor", "value",
                              DATATYPE_INTEGER,
                              true,
                              0, Integer.MAX_VALUE);

            // root -> Transparency -> TileTransparencies
            impl.addElement("TileTransparencies", "Transparency",
                            0, Integer.MAX_VALUE);

            // root -> Transparency -> TransparentTile
            impl.addElement("TransparentTile", "TileTransparencies",
                            CHILD_POLICY_EMPTY);
            impl.addAttribute("TransparentTile", "x",
                              DATATYPE_INTEGER,
                              true,
                              null);
            impl.addAttribute("TransparentTile", "y",
                              DATATYPE_INTEGER,
                              true,
                              null);

            // root -> Transparency -> TileOpacities
            impl.addElement("TileOpacities", "Transparency",
                            0, Integer.MAX_VALUE);

            // root -> Transparency -> OpaqueTile
            impl.addElement("OpaqueTile", "TileOpacities",
                            CHILD_POLICY_EMPTY);
            impl.addAttribute("OpaqueTile", "x",
                              DATATYPE_INTEGER,
                              true,
                              null);
            impl.addAttribute("OpaqueTile", "y",
                              DATATYPE_INTEGER,
                              true,
                              null);

            standardFormat = impl;
        }
    }

    /**
     * Returns an <code>IIOMetadataFormat</code> object describing the
     * standard, plug-in neutral <code>com.sun.imagei_1.0o</code>
     * metadata document format described in the main comment of the
     * <code>IIOMetadata</code> interface.
     *
     * @return a predefined <code>IIOMetadataFormat</code> instance.
     *
     * @see IIOMetadata
     */
    public static IIOMetadataFormat getStandardFormatInstance() {
        createStandardFormat();
        return standardFormat;
    }

    private Element getElement(String elementName) {
        Element element = (Element)elementMap.get(elementName);
        if (element == null) {
            throw new IllegalArgumentException("No such element \"" +
                                               elementName + "\"!");
        }
        return element;
    }

    // Setup

    /**
     * Adds a new element type to this metadata document format with a
     * child policy other than <code>CHILD_POLICY_REPEAT</code>.
     *
     * @param elementName the name of the new element.
     * @param parentName the name of the element that will be the
     * parent of the new element.
     * @param childPolicy one of the <code>CHILD_POLICY_*</code>
     * constants, other than <code>CHILD_POLICY_REPEAT</code>,
     * indicating the child policy of the new element.
     *
     * @exception IllegalArgumentException if <code>parentName</code>
     * is <code>null</code>, or is not a legal element name for this
     * format.
     * @exception IllegalArgumentException if <code>childPolicy</code>
     * is not one of the predefined constants.
     */
    protected void addElement(String elementName,
                              String parentName,
                              int childPolicy) {
        Element parent = (Element)elementMap.get(parentName);
        if (parent == null) {            
            throw new IllegalArgumentException("Invalid parent name!");
        }
        if (childPolicy < CHILD_POLICY_EMPTY ||
            childPolicy >= CHILD_POLICY_REPEAT) {
            throw new IllegalArgumentException("Invalid value for childPolicy!");
        }

        Element element = new Element();
        element.elementName = elementName;
        element.parentName = parentName;
        element.childPolicy = childPolicy;

        parent.childList.add(elementName);

        elementMap.put(elementName, element);
    }

    /**
     * Adds a new element type to this metadata document format with a
     * child policy of <code>CHILD_POLICY_REPEAT</code>.
     *
     * @param elementName the name of the new element.
     * @param parentName the name of the element that will be the
     * parent of the new element.
     * @param minChildren the minimum number of children of the node.
     * @param maxChildren the maximum number of children of the node.
     *
     * @exception IllegalArgumentException if <code>parentName</code>
     * is <code>null</code>, or is not a legal element name for this
     * format.
     * @exception IllegalArgumentException if <code>minChildren</code>
     * is negative or larger than <code>maxChildren</code>.
     */
    protected void addElement(String elementName,
                              String parentName,
                              int minChildren,
                              int maxChildren) {
        Element parent = (Element)elementMap.get(parentName);
        if (parent == null) {            
            throw new IllegalArgumentException("Invalid parent name!");
        }
        if (minChildren < 0) {
            throw new IllegalArgumentException("minChildren < 0!");
        }
        if (minChildren > maxChildren) {
            throw new IllegalArgumentException("minChildren > maxChildren!");
        }

        Element element = new Element();
        element.elementName = elementName;
        element.parentName = parentName;
        element.childPolicy = CHILD_POLICY_REPEAT;
        element.minChildren = minChildren;
        element.maxChildren = maxChildren;

        parent.childList.add(elementName);

        elementMap.put(elementName, element);
    }

    /**
     * Removes an element from the format.  If no element with the
     * given name was present, nothing happens and no exception is
     * thrown.
     *
     * @param elementName the name of the element to be removed.
     */
    protected void removeElement(String elementName) {
        elementMap.remove(elementName);
    }

    /**
     * Adds a new attribute to a previously defined element that may
     * be set to an arbitrary value.
     *
     * @param elementName the name of the element.
     * @param attrName the name of the attribute being added.
     * @param dataType the data type (string format) of the attribute,
     * one of the <code>DATATYPE_*</code> constants.
     * @param required <code>true</code> if the attribute must be present.
     * @param defaultValue the default value for the attribute, or
     * <code>null</code>.
     *
     * @exception IllegalArgumentException if <code>elementName</code>
     * is <code>null</code>, or is not a legal element name for this
     * format.
     * @exception IllegalArgumentException if <code>dataType</code> is
     * not one of the predefined constants.
     */
    protected void addAttribute(String elementName,
                                String attrName,
                                int dataType,
                                boolean required,
                                String defaultValue) {
        Element element = getElement(elementName);
        if (element == null) {
            throw new IllegalArgumentException("Invalid element name!");
        }
        if (dataType < DATATYPE_STRING || dataType > DATATYPE_TIME) {
            throw new IllegalArgumentException("Invalid value for dataType!");
        }

        Attribute attr = new Attribute();
        attr.attrName = attrName;
        attr.valueType = VALUE_ARBITRARY;
        attr.dataType = dataType;
        attr.required = required;
        attr.defaultValue = defaultValue;

        element.attrList.add(attrName);
        element.attrMap.put(attrName, attr);
    }

    /**
     * Adds a new attribute to a previously defined element that will
     * be defined by a set of enumerated values.
     *
     * @param elementName the name of the element.
     * @param attrName the name of the attribute being added.
     * @param dataType the data type (string format) of the attribute,
     * one of the <code>DATATYPE_*</code> constants.
     * @param required <code>true</code> if the attribute must be present.
     * @param defaultValue the default value for the attribute, or
     * <code>null</code>.
     * @param enumeratedValues a <code>List</code> of
     * <code>String</code>s containing the legal values for the
     * attribute.
     *
     * @exception IllegalArgumentException if <code>elementName</code>
     * is <code>null</code>, or is not a legal element name for this
     * format.
     * @exception IllegalArgumentException if <code>dataType</code> is
     * not one of the predefined constants.
     * @exception IllegalArgumentException if
     * <code>enumeratedValues</code> is <code>null</code>.
     * @exception IllegalArgumentException if
     * <code>enumeratedValues</code> does not contain at least one
     * entry.
     * @exception IllegalArgumentException if
     * <code>enumeratedValues</code> contains an element that is not a
     * <code>String</code> or is <code>null</code>.
     */
    protected void addAttribute(String elementName,
                                String attrName,
                                int dataType,
                                boolean required,
                                String defaultValue,
                                List enumeratedValues) {
        Element element = getElement(elementName);
        if (element == null) {
            throw new IllegalArgumentException("Invalid element name!");
        }
        if (dataType < DATATYPE_STRING || dataType > DATATYPE_TIME) {
            throw new IllegalArgumentException("Invalid value for dataType!");
        }
        if (enumeratedValues == null) {
            throw new IllegalArgumentException("enumeratedValues == null!");
        }
        if (enumeratedValues.size() == 0) {
            throw new IllegalArgumentException("enumeratedValues is empty!");
        }
        Iterator iter = enumeratedValues.iterator();
        while (iter.hasNext()) {
            Object o = iter.next();
            if (o == null) {
                throw new IllegalArgumentException("enumeratedValues contains a null!");
            }
            if (!(o instanceof String)) {
                throw new IllegalArgumentException("enumeratedValues contains a value that is not a String!");
            }
        }

        Attribute attr = new Attribute();
        attr.attrName = attrName;
        attr.valueType = VALUE_ENUMERATION;
        attr.dataType = dataType;
        attr.required = required;
        attr.defaultValue = defaultValue;
        attr.enumeratedValues = enumeratedValues;

        element.attrList.add(attrName);
        element.attrMap.put(attrName, attr);
    }

    /**
     * Adds a new attribute to a previously defined element that will
     * be defined by a range of values.
     *
     * @param elementName the name of the element.
     * @param attrName the name of the attribute being added.
     * @param dataType the data type (string format) of the attribute,
     * one of the <code>DATATYPE_*</code> constants.
     * @param required <code>true</code> if the attribute must be present.
     * @param defaultValue the default value for the attribute, or
     * <code>null</code>.
     * @param minValue the smallest (inclusive or exclusive depending
     * on the value of <code>minInclusive</code>) legal value for the
     * attribute, as a <code>String</code>.
     * @param maxValue the largest (inclusive or exclusive depending
     * on the value of <code>minInclusive</code>) legal value for the
     * attribute, as a <code>String</code>.
     * @param minInclusive <code>true</code> if <code>minValue</code>
     * is inclusive.
     * @param maxInclusive <code>true</code> if <code>maxValue</code>
     * is inclusive.
     *
     * @exception IllegalArgumentException if <code>elementName</code>
     * is <code>null</code>, or is not a legal element name for this
     * format.
     * @exception IllegalArgumentException if <code>dataType</code> is
     * not one of the predefined constants.
     */
    protected void addAttribute(String elementName,
                                String attrName,
                                int dataType,
                                boolean required,
                                String defaultValue,
                                String minValue,
                                String maxValue,
                                boolean minInclusive,
                                boolean maxInclusive) {
        Element element = getElement(elementName);
        if (element == null) {
            throw new IllegalArgumentException("Invalid element name!");
        }
        if (dataType < DATATYPE_STRING || dataType > DATATYPE_TIME) {
            throw new IllegalArgumentException("Invalid value for dataType!");
        }

        Attribute attr = new Attribute();
        attr.attrName = attrName;
        attr.valueType = VALUE_RANGE;
        if (minInclusive) {
            attr.valueType |= VALUE_RANGE_MIN_INCLUSIVE_MASK;
        }
        if (maxInclusive) {
            attr.valueType |= VALUE_RANGE_MAX_INCLUSIVE_MASK;
        }
        attr.dataType = dataType;
        attr.required = required;
        attr.defaultValue = defaultValue;
        attr.minValue = minValue;
        attr.maxValue = maxValue;

        element.attrList.add(attrName);
        element.attrMap.put(attrName, attr);
    }

    /**
     * Adds a new attribute to a previously defined element that will
     * be defined by a list of values.
     *
     * @param elementName the name of the element.
     * @param attrName the name of the attribute being added.
     * @param dataType the data type (string format) of the attribute,
     * one of the <code>DATATYPE_*</code> constants.
     * @param required <code>true</code> if the attribute must be present.
     * @param listMinLength the smallest legal number of list items.
     * @param listMaxLength the largest legal number of list items.
     *
     * @exception IllegalArgumentException if <code>elementName</code>
     * is <code>null</code>, or is not a legal element name for this
     * format.
     * @exception IllegalArgumentException if <code>dataType</code> is
     * not one of the predefined constants.
     * @exception IllegalArgumentException if
     * <code>listMinLength</code> is negative or larger than
     * <code>listMaxLength</code>.
     */
    protected void addAttribute(String elementName,
                                String attrName,
                                int dataType,
                                boolean required,
                                int listMinLength,
                                int listMaxLength) {
        Element element = getElement(elementName);
        if (element == null) {
            throw new IllegalArgumentException("Invalid element name!");
        }
        if (dataType < DATATYPE_STRING || dataType > DATATYPE_TIME) {
            throw new IllegalArgumentException("Invalid value for dataType!");
        }
        if (listMinLength < 0 || listMinLength > listMaxLength) {
            throw new IllegalArgumentException("Invalid list bounds!");
        }

        Attribute attr = new Attribute();
        attr.attrName = attrName;
        attr.valueType = VALUE_LIST;
        attr.dataType = dataType;
        attr.required = required;
        attr.listMinLength = listMinLength;
        attr.listMaxLength = listMaxLength;

        element.attrList.add(attrName);
        element.attrMap.put(attrName, attr);
    }

    /**
     * Adds a new attribute to a previously defined element that will
     * be defined by the enumerated values <code>TRUE</code> and
     * <code>FALSE</code>, with a datatype of
     * <code>DATATYPE_BOOLEAN</code>.
     *
     * @param elementName the name of the element.
     * @param attrName the name of the attribute being added.
     * @param hasDefaultValue <code>true</code> if a default value
     * should be present.
     * @param defaultValue the default value for the attribute as a
     * <code>boolean</code>, ignored if <code>hasDefaultValue</code>
     * is <code>false</code>.
     *
     * @exception IllegalArgumentException if <code>elementName</code>
     * is <code>null</code>, or is not a legal element name for this
     * format.
     */
    protected void addBooleanAttribute(String elementName,
                                       String attrName,
                                       boolean hasDefaultValue,
                                       boolean defaultValue) {
        List values = new ArrayList();
        values.add("TRUE");
        values.add("FALSE");
        
        String dval = null;
        if (hasDefaultValue) {
            dval = defaultValue ? "TRUE" : "FALSE";
        }
        addAttribute(elementName,
                     attrName,
                     DATATYPE_BOOLEAN,
                     true,
                     dval,
                     values);
    }

    /**
     * Removes an attribute from a previously defined element.  If no
     * attribute with the given name was present in the given element,
     * nothing happens and no exception is thrown.
     *
     * @param elementName the name of the element.
     * @param attrName the name of the attribute being removed.
     *
     * @exception IllegalArgumentException if <code>elementName</code>
     * is <code>null</code>, or is not a legal element name for this format.
     */
    protected void removeAttribute(String elementName, String attrName) {
        Element element = getElement(elementName);
        if (element == null) {
            throw new IllegalArgumentException("Invalid element name!");
        }

        element.attrList.remove(attrName);
        element.attrMap.remove(attrName);
    }

    /**
     * Allows an <code>Object</code> reference of a given class type
     * to be stored in nodes implementing the named element.  The
     * value of the <code>Object</code> is unconstrained other than by
     * its class type.
     *
     * <p> If an <code>Object</code> reference was previously allowed,
     * the previous settings are overwritten.
     *
     * @param elementName the name of the element.
     * @param classType a <code>Class</code> variable indicating the
     * legal class type for the object value.
     * @param required <code>true</code> if an object value must be present.
     * @param defaultValue the default value for the
     * <code>Object</code> reference, or <code>null</code>.
     * 
     * @exception IllegalArgumentException if <code>elementName</code>
     * is <code>null</code>, or is not a legal element name for this format.
     */
    protected void addObjectValue(String elementName,
                                  Class classType,
                                  boolean required,
                                  Object defaultValue) {
        Element element = getElement(elementName);
        if (element == null) {
            throw new IllegalArgumentException("Invalid element name!");
        }

        ObjectValue obj = new ObjectValue();
        obj.valueType = VALUE_ARBITRARY;
        obj.classType = classType;
        obj.defaultValue = defaultValue;

        element.objectValue = obj;
    }

    /**
     * Allows an <code>Object</code> reference of a given class type
     * to be stored in nodes implementing the named element.  The
     * value of the <code>Object</code> must be one of the values
     * given by <code>enumeratedValues</code>.
     *
     * <p> If an <code>Object</code> reference was previously allowed,
     * the previous settings are overwritten.
     *
     * @param elementName the name of the element.
     * @param classType a <code>Class</code> variable indicating the
     * legal class type for the object value.
     * @param required <code>true</code> if an object value must be present.
     * @param defaultValue the default value for the
     * <code>Object</code> reference, or <code>null</code>.
     * @param enumeratedValues a <code>List</code> of
     * <code>Object</code>s containing the legal values for the
     * object reference.
     * 
     * @exception IllegalArgumentException if <code>elementName</code>
     * is <code>null</code>, or is not a legal element name for this format.
     * @exception IllegalArgumentException if
     * <code>enumeratedValues</code> is <code>null</code>.
     * @exception IllegalArgumentException if
     * <code>enumeratedValues</code> does not contain at least one
     * entry.
     * @exception IllegalArgumentException if
     * <code>enumeratedValues</code> contains an element that is not
     * an instance of the class type denoted by <code>classType</code>
     * or is <code>null</code>.
     */
    protected void addObjectValue(String elementName,
                                  Class classType,
                                  boolean required,
                                  Object defaultValue,
                                  List enumeratedValues) {
        Element element = getElement(elementName);
        if (element == null) {
            throw new IllegalArgumentException("Invalid element name!");
        }
        if (enumeratedValues == null) {
            throw new IllegalArgumentException("enumeratedValues == null!");
        }
        if (enumeratedValues.size() == 0) {
            throw new IllegalArgumentException("enumeratedValues is empty!");
        }
        Iterator iter = enumeratedValues.iterator();
        while (iter.hasNext()) {
            Object o = iter.next();
            if (o == null) {
                throw new IllegalArgumentException("enumeratedValues contains a null!");
            }
            if (!classType.isInstance(o)) {
                throw new IllegalArgumentException("enumeratedValues contains a value not of class classType!");
            }
        }

        ObjectValue obj = new ObjectValue();
        obj.valueType = VALUE_ENUMERATION;
        obj.classType = classType;
        obj.defaultValue = defaultValue;
        obj.enumeratedValues = enumeratedValues;

        element.objectValue = obj;
    }

    /**
     * Allows an <code>Object</code> reference of a given class type
     * to be stored in nodes implementing the named element.  The
     * value of the <code>Object</code> must be within the range given
     * by <code>minValue</code> and <code>maxValue</code>.
     * Furthermore, the class type must implement the
     * <code>Comparable</code> interface.
     *
     * <p> If an <code>Object</code> reference was previously allowed,
     * the previous settings are overwritten.
     *
     * @param elementName the name of the element.
     * @param classType a <code>Class</code> variable indicating the
     * legal class type for the object value.
     * @param defaultValue the default value for the
     * @param minValue the smallest (inclusive or exclusive depending
     * on the value of <code>minInclusive</code>) legal value for the
     * object value, as a <code>String</code>.
     * @param maxValue the largest (inclusive or exclusive depending
     * on the value of <code>minInclusive</code>) legal value for the
     * object value, as a <code>String</code>. 
     * @param minInclusive <code>true</code> if <code>minValue</code>
     * is inclusive.
     * @param maxInclusive <code>true</code> if <code>maxValue</code>
     * is inclusive.
     *
     * @exception IllegalArgumentException if <code>elementName</code>
     * is <code>null</code>, or is not a legal element name for this
     * format.
     */
    protected void addObjectValue(String elementName,
                                  Class classType,
                                  Object defaultValue,
                                  Comparable minValue,
                                  Comparable maxValue,
                                  boolean minInclusive,
                                  boolean maxInclusive) {
        Element element = getElement(elementName);
        if (element == null) {
            throw new IllegalArgumentException("Invalid element name!");
        }

        ObjectValue obj = new ObjectValue();
        obj.valueType = VALUE_RANGE;
        if (minInclusive) {
            obj.valueType |= VALUE_RANGE_MIN_INCLUSIVE_MASK;
        }
        if (maxInclusive) {
            obj.valueType |= VALUE_RANGE_MAX_INCLUSIVE_MASK;
        }
        obj.classType = classType;
        obj.defaultValue = defaultValue;
        obj.minValue = minValue;
        obj.maxValue = maxValue;

        element.objectValue = obj;
    }

    /**
     * Allows an <code>Object</code> reference of a given class type
     * to be stored in nodes implementing the named element.  The
     * value of the <code>Object</code> must an array of objects of
     * class type given by <code>classType</code>, with at least
     * <code>arrayMinLength</code> and at most
     * <code>arrayMaxLength</code> elements.
     *
     * <p> If an <code>Object</code> reference was previously allowed,
     * the previous settings are overwritten.
     *
     * @param elementName the name of the element.
     * @param classType a <code>Class</code> variable indicating the
     * legal class type for the object value.
     * @param arrayMinLength the smallest legal length for the array.
     * @param arrayMaxLength the largest legal length for the array.
     *
     * @exception IllegalArgumentException if <code>elementName</code> is
     * not a legal element name for this format.
     */
    protected void addObjectValue(String elementName,
                                  Class classType,
                                  int arrayMinLength,
                                  int arrayMaxLength) {
        Element element = getElement(elementName);
        if (element == null) {
            throw new IllegalArgumentException("Invalid element name!");
        }

        ObjectValue obj = new ObjectValue();
        obj.valueType = VALUE_LIST;
        obj.classType = classType;
        obj.arrayMinLength = arrayMinLength;
        obj.arrayMaxLength = arrayMaxLength;

        element.objectValue = obj;
    }

    /**
     * Disallows an <code>Object</code> reference from being stored in
     * nodes implementing the named element.
     *
     * @param elementName the name of the element.
     * 
     * @exception IllegalArgumentException if <code>elementName</code> is
     * not a legal element name for this format.
     */
    protected void removeObjectValue(String elementName) {
        Element element = getElement(elementName);
        if (element == null) {
            throw new IllegalArgumentException("Invalid element name!");
        }

        element.objectValue = null;
    }

    // Utility method

    // Methods from IIOMetadataFormat

    // Root

    public String getRootName() {
        return rootName;
    }

    // Parents

    public String getParentName(String elementName) {
        Element element = getElement(elementName);
        return element.parentName;
    }

    // Multiplicity

    public boolean canNodeAppear(String elementName,
                                 ImageTypeSpecifier imageType) {
        return true;
    }

    public int getElementMinChildren(String elementName) {
        Element element = getElement(elementName);
        if (element.childPolicy != CHILD_POLICY_REPEAT) {
            throw new IllegalArgumentException("Child policy not CHILD_POLICY_REPEAT!");
        }
        return element.maxChildren;
    }

    public int getElementMaxChildren(String elementName) {
        Element element = getElement(elementName);
        if (element.childPolicy != CHILD_POLICY_REPEAT) {
            throw new IllegalArgumentException("Child policy not CHILD_POLICY_REPEAT!");
        }
        return element.maxChildren;
    }

    // Children

    public int getChildPolicy(String elementName) {
        Element element = getElement(elementName);
        return element.childPolicy;
    }

    public String[] getChildNames(String elementName) {
        Element element = getElement(elementName);
        return (String[])element.childList.toArray(new String[0]);
    }

    // Attributes

    public String[] getAttributeNames(String elementName) {
        Element element = getElement(elementName);
        List names = element.attrList;

        String[] result = new String[names.size()];
        return (String[])names.toArray(result);
    }

    private Attribute getAttribute(String elementName, String attrName) {
        Element element = getElement(elementName);
        Attribute attr = (Attribute)element.attrMap.get(attrName);
        return attr;
    }

    public int getAttributeValueType(String elementName, String attrName) {
        Attribute attr = getAttribute(elementName, attrName);
        return attr.valueType;
    }
    
    public int getAttributeDataType(String elementName, String attrName) { 
        Attribute attr = getAttribute(elementName, attrName);
        return attr.dataType;
    }

    public boolean isAttributeRequired(String elementName, String attrName) {
        Attribute attr = getAttribute(elementName, attrName);
        return attr.required;
    }

    public String getAttributeDefaultValue(String elementName,
                                           String attrName) {
        Attribute attr = getAttribute(elementName, attrName);
        return attr.defaultValue;
    }

    public String[] getAttributeEnumerations(String elementName,
                                             String attrName) {
        Attribute attr = getAttribute(elementName, attrName);
        if (attr.valueType != VALUE_ENUMERATION) {
            throw new IllegalStateException("Attribute not an enumeration!");
        }
        
        List values = attr.enumeratedValues;
        Iterator iter = values.iterator();
        String[] result = new String[values.size()];
        return (String[])values.toArray(result);
    }

    public String getAttributeMinValue(String elementName, String attrName) {
        Attribute attr = getAttribute(elementName, attrName);
        if (attr.valueType != VALUE_RANGE) {
            throw new IllegalStateException("Attribute not a range!");
        }

        return attr.minValue;
    }

    public String getAttributeMaxValue(String elementName, String attrName) {
        Attribute attr = getAttribute(elementName, attrName);
        if (attr.valueType != VALUE_RANGE) {
            throw new IllegalStateException("Attribute not a range!");
        }

        return attr.maxValue;
    }

    public int getAttributeListMinLength(String elementName, String attrName) {
        Attribute attr = getAttribute(elementName, attrName);
        if (attr.valueType != VALUE_LIST) {
            throw new IllegalStateException("Attribute not a list!");
        }

        return attr.listMinLength;
    }

    public int getAttributeListMaxLength(String elementName, String attrName) {
        Attribute attr = getAttribute(elementName, attrName);
        if (attr.valueType != VALUE_LIST) {
            throw new IllegalStateException("Attribute not a list!");
        }

        return attr.listMaxLength;
    }

    private ObjectValue getObjectValue(String elementName) {
        Element element = getElement(elementName);
        ObjectValue objv = (ObjectValue)element.objectValue;
        if (objv == null) {
            throw new IllegalArgumentException("No object within element " +
                                               elementName + "!");
        }
        return objv;
    }

    public int getObjectValueType(String elementName) {
        Element element = getElement(elementName);
        ObjectValue objv = (ObjectValue)element.objectValue;
        if (objv == null) {
            return VALUE_NONE;
        }
        return objv.valueType;
    }

    public Class getObjectClass(String elementName) {
        ObjectValue objv = getObjectValue(elementName);
        return objv.classType;
    }

    public Object getObjectDefaultValue(String elementName) {
        ObjectValue objv = getObjectValue(elementName);
        return objv.defaultValue;
    }

    public Object[] getObjectEnumerations(String elementName) {
        ObjectValue objv = getObjectValue(elementName);
        if (objv.valueType != VALUE_ENUMERATION) {
            throw new IllegalArgumentException("Not an enumeration!");
        }
        List vlist = objv.enumeratedValues;
        Object[] values = new Object[vlist.size()];
        return vlist.toArray(values);
    }

    public Comparable getObjectMinValue(String elementName) {
        ObjectValue objv = getObjectValue(elementName);
        if (objv.valueType != VALUE_RANGE) {
            throw new IllegalArgumentException("Not a range!");
        }
        return objv.minValue;
    }

    public Comparable getObjectMaxValue(String elementName) {
        ObjectValue objv = getObjectValue(elementName);
        if (objv.valueType != VALUE_RANGE) {
            throw new IllegalArgumentException("Not a range!");
        }
        return objv.maxValue;
    }

    public int getObjectArrayMinLength(String elementName) {
        ObjectValue objv = getObjectValue(elementName);
        if (objv.valueType != VALUE_LIST) {
            throw new IllegalArgumentException("Not a list!");
        }
        return objv.arrayMinLength;
    }

    public int getObjectArrayMaxLength(String elementName) {
        ObjectValue objv = getObjectValue(elementName);
        if (objv.valueType != VALUE_LIST) {
            throw new IllegalArgumentException("Not a list!");
        }
        return objv.arrayMinLength;
    }

//      private void indent(StringBuffer s, int level) {
//          for (int i = 0; i < level; i++) {
//              s.append("  ");
//          }
//      }

//      private static final String[] datatypeNames = {
//          "string", "boolean", "integer", "float", "double",
//          "date", "time"
//      };

//      private void appendElement(StringBuffer s, String elementName, int level) {
//          int childPolicy = getChildPolicy(elementName);
        
//          s.append("\n\n");
//          indent(s, level);
//          s.append("<!ELEMENT " + elementName);
        
//          String[] childNames = getChildNames(elementName);
//          if (childPolicy == CHILD_POLICY_EMPTY) {
//              s.append(" EMPTY>");
//          } else {
//              for (int i = 0; i < childNames.length; i++) {
//                  if (i == 0) {
//                      s.append("\n");
//                      indent(s, level);
//                      s.append("  (");
//                  } else {
//                      indent(s, level);
//                      s.append("   ");
//                  }
//                  s.append(childNames[i]);
//                  if (childPolicy == CHILD_POLICY_SOME) {
//                      s.append("?");
//                  } else if (childPolicy == CHILD_POLICY_REPEAT) {
//                      s.append("*");
//                  }
//                  if (i < childNames.length - 1) {
//                      if (childPolicy == CHILD_POLICY_CHOICE) {
//                          s.append("|\n");
//                      } else {
//                          s.append(",\n");
//                      }
//                  } else {
//                      s.append(")");
//                  }
//              }
//              s.append(">");
//          }
        
//          // Attributes
//          String[] attrNames = getAttributeNames(elementName);
//          for (int i = 0; i < attrNames.length; i++) {
//              String attrName = attrNames[i];

//              s.append("\n");
//              indent(s, level + 1);
//              s.append("<!ATTLIST ");
//              s.append(elementName);
//              s.append(" ");
//              s.append(attrName);

//              int attrValueType = getAttributeValueType(elementName, attrName);
//              int datatype = -1;
//              boolean isList = false;
//              int listMinLength = -1;
//              int listMaxLength = -1;

//              if (attrValueType == VALUE_ENUMERATION) {
//                  String[] enums =
//                      getAttributeEnumerations(elementName, attrName);
//                  s.append("\n");
//                  indent(s, level + 2);
//                  s.append("(");
//                  for (int j = 0; j < enums.length; j++) {
//                      if (j > 0) {
//                          s.append(" |\n");
//                          indent(s, level + 2);
//                          s.append(" ");
//                      }
//                      s.append(enums[j]);
//                  }
//                  s.append(")");
//              } else {
//                  datatype = getAttributeDataType(elementName, attrName);
//                  if (attrValueType == VALUE_LIST) {
//                      isList = true;
//                      listMinLength =
//                          getAttributeListMinLength(elementName, attrName);
//                      listMaxLength =
//                          getAttributeListMaxLength(elementName, attrName);
//                  }
//                  s.append(" CDATA");
//              }

//              String defaultValue =
//                  getAttributeDefaultValue(elementName, attrName);
//              if (defaultValue != null) {
//                  s.append(" \"");
//                  s.append(defaultValue);
//                  s.append("\"");
//              } else if (isAttributeRequired(elementName, attrName)) {
//                  s.append(" #REQUIRED");
//              } else {
//                  s.append(" #IMPLIED");
//              }
//              s.append(">");

//              if (datatype != -1) {
//                  s.append(" <!-- ");
//                  if (isList) {
//                      s.append("list of ");
//                  }
//                  s.append(datatypeNames[datatype]);
//                  s.append(" -->");
//              }
//          }
        
//          // Object value
        
//          // Children
//          for (int i = 0; i < childNames.length; i++) {
//              appendElement(s, childNames[i], level + 1);
//          }
//      }

//      // Creates a DTD with comments
//      public String toString() {
//          StringBuffer s = new StringBuffer();

//          s.append("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n");
//          s.append("<!DOCTYPE " + getRootName() + " [");

//          appendElement(s, getRootName(), 1);
        
//          return s.toString();
//      }

//      public static void main(String[] args) {
//          System.out.println(getStandardFormatInstance().toString());
//      }
}
