/******************************************************************************/
/* (c) 2005 jamie twycross, jpt AT cs.nott.ac.uk                              */
/* released under gnu gpl v2                                                  */
/* cell api                                                                   */
/******************************************************************************/

#include "cell.h"
#include "response.h"

#include <stdlib.h>
#include <err.h>

/* initialise a cell */
Cell *init_cell(Cell_Param *cell_param)
{
	Cell *cell;
	unsigned int i;

	if(!(cell = malloc(sizeof(Cell))))
		err(1, "couldn't allocate memory for cell");

	/* parameters */
	cell->id = cell_param->id;
	cell->type = cell_param->type;
	cell->state = cell_param->state;
	cell->iterations = 0;
	cell->cycle = cell_param->cycle;

	/* internal storage */
	cell->num_antigen = cell_param->num_antigen;
	if(cell->num_antigen) {
		if(!(cell->antigen = calloc(cell->num_antigen, sizeof(Antigen *))))
			err(1, "couldn't allocate memory for antigen");
	}
	cell->num_cytokines = cell_param->num_cytokines;
	if(cell->num_cytokines) {
		if(!(cell->cytokine = calloc(cell->num_cytokines, sizeof(Cytokine *))))
			err(1, "couldn't allocate memory for cytokines");
		for(i = 0; i < cell->num_cytokines; i++) {
			cell->cytokine[i] = init_cytokine();
			cell->cytokine[i]->id = i;
		}
	}

	/* input receptors */
	cell->num_antigen_receptors = cell_param->num_antigen_receptors;
	if(cell->num_antigen_receptors) {
		if(!(cell->antigen_receptor = calloc(cell->num_antigen_receptors, \
				sizeof(Antigen_Receptor *))))
			err(1, "couldn't allocate memory for antigen receptors");
		for(i = 0; i < cell->num_antigen_receptors; i++)
			cell->antigen_receptor[i] = init_antigen_receptor();
	}
	cell->num_cytokine_receptors = cell_param->num_cytokine_receptors;
	if(cell->num_cytokine_receptors) {
		if(!(cell->cytokine_receptor = calloc(cell->num_cytokine_receptors, \
				sizeof(Cytokine_Receptor *))))
			err(1, "couldn't allocate memory for cytokine receptors");
		for(i = 0; i < cell->num_cytokine_receptors; i++) {
			cell->cytokine_receptor[i] = init_cytokine_receptor();
			cell->cytokine_receptor[i]->id = i;
		}
	}
	cell->num_cell_receptors = cell_param->num_cell_receptors;
	if(cell->num_cell_receptors) {
		if(!(cell->cell_receptor = calloc(cell->num_cell_receptors, \
				sizeof(Cell_Receptor *))))
			err(1, "couldn't allocate memory for cell receptors");
		for(i = 0; i < cell->num_cell_receptors; i++)
			cell->cell_receptor[i] = init_cell_receptor();
	}
	cell->num_vr_receptors = cell_param->num_vr_receptors;
	if(cell->num_vr_receptors) {
		if(!(cell->vr_receptor = calloc(cell->num_vr_receptors, \
				sizeof(VR_Receptor *))))
			err(1, "couldn't allocate memory for variable region receptors");
		for(i = 0; i < cell->num_vr_receptors; i++)
			cell->vr_receptor[i] = init_vr_receptor();
	}

	/* output producers */
	cell->num_cytokine_producers = cell_param->num_cytokine_producers;
	if(cell->num_cytokine_producers) {
		if(!(cell->cytokine_producer = calloc(cell->num_cytokine_producers, \
				sizeof(Cytokine_Producer *))))
			err(1, "couldn't allocate memory for cytokine producers");
		for(i = 0; i < cell->num_cytokine_producers; i++) {
			cell->cytokine_producer[i] = init_cytokine_producer();
			cell->cytokine_producer[i]->type = i;
		}
	}
	cell->num_antigen_producers = cell_param->num_antigen_producers;
	if(cell->num_antigen_producers) {
		if(!(cell->antigen_producer = calloc(cell->num_antigen_producers, \
				sizeof(Antigen_Producer *))))
			err(1, "couldn't allocate memory for antigen producers");
		for(i = 0; i < cell->num_antigen_producers; i++)
			cell->antigen_producer[i] = init_antigen_producer();
	}
	cell->num_response_producers = cell_param->num_response_producers;
	if(cell->num_response_producers) {
		if(!(cell->response_producer = calloc(cell->num_response_producers, \
				sizeof(Response_Producer *))))
			err(1, "couldn't allocate memory for response producers");
		for(i = 0; i < cell->num_response_producers; i++)
			cell->response_producer[i] = init_response_producer();
	}

	return cell;
}

void free_cell(Cell *cell)
{
	unsigned int i;

	if(!cell)
		return;

	/* internal storage */
	if(cell->num_antigen) {
		for(i = 0; i < cell->num_antigen; i++)
			free_antigen(cell->antigen[i]);
		free(cell->antigen);
	}
	if(cell->num_cytokines) {
		for(i = 0; i < cell->num_cytokines; i++)
			free_cytokine(cell->cytokine[i]);
		free(cell->cytokine);
	}

	/* input receptors */
	if(cell->num_antigen_receptors) {
		for(i = 0; i < cell->num_antigen_receptors; i++)
			free_antigen_receptor(cell->antigen_receptor[i]);
		free(cell->antigen_receptor);
	}
	if(cell->num_cytokine_receptors) {
		for(i = 0; i < cell->num_cytokine_receptors; i++)
			free_cytokine_receptor(cell->cytokine_receptor[i]);
		free(cell->cytokine_receptor);
	}
	if(cell->num_cell_receptors) {
		for(i = 0; i < cell->num_cell_receptors; i++)
			free_cell_receptor(cell->cell_receptor[i]);
		free(cell->cell_receptor);
	}
	if(cell->num_vr_receptors) {
		for(i = 0; i < cell->num_vr_receptors; i++)
			free_vr_receptor(cell->vr_receptor[i]);
		free(cell->vr_receptor);
	}

	/* output producers */
	if(cell->num_cytokine_producers) {
		for(i = 0; i < cell->num_cytokine_producers; i++)
			free_cytokine_producer(cell->cytokine_producer[i]);
		free(cell->cytokine_producer);
	}
	if(cell->num_antigen_producers) {
		for(i = 0; i < cell->num_antigen_producers; i++)
			free_antigen_producer(cell->antigen_producer[i]);
		free(cell->antigen_producer);
	}
	if(cell->num_response_producers) {
		for(i = 0; i < cell->num_response_producers; i++)
			free_response_producer(cell->response_producer[i]);
		free(cell->response_producer);
	}

	free(cell);
}

void print_cell(Cell *cell)
{
	unsigned int i, j, count;

	if(!cell)
		return;

	printf("cell: id=%d type=%d state=%d nar=%d nckr=%d ncr=%d nvrr=%d" \
			" nckp=%d nap=%d nrp=%d", \
			cell->id, cell->type, cell->state, cell->num_antigen_receptors, \
			cell->num_cytokine_receptors, cell->num_cell_receptors, \
			cell->num_vr_receptors, cell->num_cytokine_producers, \
			cell->num_antigen_producers, cell->num_response_producers);
	if(cell->num_antigen) {
		printf(" antigen:");
		count = 0;
		for(i = 0; i < cell->num_antigen; i++)
			if(cell->antigen[i]) {
				printf(" (%d,", cell->antigen[i]->id);
				fwrite((void *) cell->antigen[i]->val, sizeof(char), \
						cell->antigen[i]->len, stdout);
				printf(")");
				count++;
			}
		printf(" ta=%d", count);
	}
	if(cell->num_cytokines) {
		printf(" cytokines:");
		for(i = 0; i < cell->num_cytokines; i++)
			printf(" (%d,%f)", cell->cytokine[i]->id, cell->cytokine[i]->val);
	}

	if(cell->num_cytokine_receptors) {
		printf(" cytokine_receptors:");
		for(i = 0; i < cell->num_cytokine_receptors; i++)
			printf(" (%d,%f)", cell->cytokine_receptor[i]->id, \
					cell->cytokine_receptor[i]->val);
	}
	if(cell->num_vr_receptors) {
		printf(" vr_receptors:");
		for(i = 0; i < cell->num_vr_receptors; i++) {
			printf(" (%d,%d,", i, cell->vr_receptor[i]->activated);
			for(j = 0; j < cell->vr_receptor[i]->lock->len; j++)
			if(cell->vr_receptor[i]->lock->val[j])
				printf("%c", cell->vr_receptor[i]->lock->val[j]);
			else
				printf("#");
			printf(",");
			if(cell->vr_receptor[i]->activated)
				fwrite((void *) cell->vr_receptor[i]->key->val, sizeof(char), \
						cell->vr_receptor[i]->key->len, stdout);
			printf(")");
		}
	}
	if(cell->num_cell_receptors) {
		printf(" cell_receptors:");
		for(i = 0; i < cell->num_cell_receptors; i++)
			if(cell->cell_receptor[i]->cell)
				printf(" (%d,%d)", i, cell->cell_receptor[i]->cell->id);
	}

	if(cell->num_cytokine_producers) {
		printf(" cytokine_producers:");
		for(i = 0; i < cell->num_cytokine_producers; i++)
			printf(" (%d,%f)", cell->cytokine_producer[i]->type, \
					cell->cytokine_producer[i]->val);
	}
	if(cell->num_antigen_producers) {
		printf(" antigen_producers:");
		for(i = 0; i < cell->num_antigen_producers; i++)
			if(cell->antigen_producer[i]->antigen) {
				printf(" (%d,", i);
				fwrite((void *) cell->antigen_producer[i]->antigen->val, \
						sizeof(char), \
						cell->antigen_producer[i]->antigen->len, stdout);
				printf(")");
			}
	}
	if(cell->num_response_producers) {
		printf(" response_producers:");
		for(i = 0; i < cell->num_response_producers; i++) {
			printf(" (%d,%d,", i, cell->response_producer[i]->type);
			if(cell->response_producer[i]->antigen)
				fwrite((void *) cell->response_producer[i]->antigen->val, \
						sizeof(char), \
						cell->response_producer[i]->antigen->len, stdout);
			printf(",");
			if(cell->response_producer[i]->antigen && \
					cell->response_producer[i]->response)
				printf("%s", cell->response_producer[i]->response);
			printf(")");
		}
	}
	printf("\n");
}

/* set the cells antigen producers */
/* remove one antigen from cell's store to antigen producer */
inline void set_antigen_producers(Cell *cell)
{
	register unsigned int i, ind;

	for(i = 0; i < cell->num_antigen_producers; i++) {
		if(cell->antigen_producer[i]->active) {
			/* currently producing antigen */
			cell->antigen_producer[i]->active--;
			continue;
		}
		/* remove any existing antigen */
		if(cell->antigen_producer[i]->antigen) {
			free_antigen(cell->antigen_producer[i]->antigen);
			cell->antigen_producer[i]->antigen = NULL;
		}
		/* get random antigen in cell */
		ind = (int) (((double) cell->num_antigen * rand()) / (RAND_MAX + 1.0));
		if(!cell->antigen[ind]) /* sampled nothing */
			continue;
		
		/* transfer probabilistically */
		if(drand48() < cell->antigen_producer[i]->efficiency) {
			/* copy the antigen from cell's store */
			cell->antigen_producer[i]->antigen = \
					dup_antigen(cell->antigen[ind]);
			if(cell->antigen_producer[i]->destroy) {
				/* remove from cell's antigen store */
				free_antigen(cell->antigen[ind]);
				cell->antigen[ind] = NULL;
			}
			/* set antigen producer as active */
			cell->antigen_producer[i]->active = \
					cell->antigen_producer[i]->action_time;
		}
	}
}

/* send any responses to response client */
inline void produce_response(Cell *cell)
{
	unsigned int i;

	if(!cell)
		return;

	/* if there's an antigen associated with response producer send response */
	for(i = 0; i < cell->num_response_producers; i++)
		if(cell->response_producer[i]->antigen)
			send_response(cell->response_producer[i]->response);
}
