/******************************************************************************/
/* (c) 2005 jamie twycross, jpt AT cs.nott.ac.uk                              */
/* released under gnu gpl v2                                                  */
/* email datastream api                                                       */
/******************************************************************************/

#include "ds.h"
#include "log.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>
#include <sys/types.h>
#include <sys/socket.h>

#define CLIENT_MSG		0x00
#define CLIENT_DS		0x01

/* initialise datastream structure */
DS *init_ds(const unsigned int num_signals, const unsigned int num_antigen)
{
	DS *ds;

	if(!(ds = malloc(sizeof(DS))))
		err(1, "couldn't allocate memory for datastream structure");
	if((ds->num_signals = num_signals))
		if(!(ds->signal = calloc(ds->num_signals, sizeof(Signal *))))
			err(1, "couldn't allocate memory for datastream signals");
	if((ds->num_antigen = num_antigen))
		if(!(ds->antigen = calloc(ds->num_antigen, sizeof(Antigen *))))
			err(1, "couldn't allocate memory for datastream antigen");

	return ds;
}

/* free datastream structure */
void free_ds(DS *ds)
{
	register unsigned int i;

	if(!ds)
		return;
	if(ds->num_signals) {
		for(i = 0; i < ds->num_signals; i++)
			free_signal(ds->signal[i]);
		free(ds->signal);
	}
	if(ds->num_antigen) {
		for(i = 0; i < ds->num_antigen; i++)
			free_antigen(ds->antigen[i]);
		free(ds->antigen);
	}
	free(ds);
}

/* read DS structure from streams */
DS *read_ds(FILE *fp)
{
	unsigned int i, num_signals, num_antigen;
	int id;
	struct timeval tm;
	DS *ds;

	/* read header */
	if(fread((void *) &tm, 1, sizeof(struct timeval), fp) != \
			sizeof(struct timeval)) {
		if(feof(fp))
			return NULL; /* EOF */
		err(1, "couldn't read ds->tm");
	}
	if(fread((void *) &id, 1, sizeof(int), fp) != sizeof(int))
		err(1, "couldn't read ds->id");
	if(fread((void *) &num_signals, 1, sizeof(unsigned int), fp) != \
			sizeof(unsigned int))
		err(1, "couldn't read ds->num_signals");
	if(fread((void *) &num_antigen, 1, sizeof(unsigned int), fp) != \
			sizeof(unsigned int))
		err(1, "couldn't read ds->num_antigen");
	/* initialise datastream structure */
	ds = init_ds(num_signals, num_antigen);
	ds->id = id;
	memcpy(&ds->tm, &tm, sizeof(struct timeval));
	/* read data + assign to datastream */
	for(i = 0; i < ds->num_signals; i++)
		ds->signal[i] = read_signal(fp);
	for(i = 0; i < ds->num_antigen; i++)
		ds->antigen[i] = read_antigen(fp);

	return ds;
}

/* write DS structure to stream */
void write_ds(DS *ds, FILE *fp)
{
	unsigned int i;

	/* write structure members */
	if(fwrite((void *) &ds->tm, 1, sizeof(struct timeval), fp) != \
			sizeof(struct timeval))
		err(1, "couldn't write ds->tm");
	if(fwrite((void *) &ds->id, 1, sizeof(int), fp) != sizeof(int))
		err(1, "couldn't write ds->id");
	if(fwrite((void *) &ds->num_signals, 1, sizeof(unsigned int), fp) != \
			sizeof(unsigned int))
		err(1, "couldn't write ds->num_signals");
	if(fwrite((void *) &ds->num_antigen, 1, sizeof(unsigned int), fp) != \
			sizeof(unsigned int))
		err(1, "couldn't write ds->num_antigen");
	for(i = 0; i < ds->num_signals; i++)
		write_signal(ds->signal[i], fp);
	for(i = 0; i < ds->num_antigen; i++)
		write_antigen(ds->antigen[i], fp);
	fflush(fp);
}

void print_ds(const DS *ds)
{
	unsigned int i;

	printf("%d.%d ds %d %d %d", (int) ds->tm.tv_sec, (int) ds->tm.tv_usec, \
			ds->id, ds->num_signals, ds->num_antigen);
	if(ds->num_signals) {
		for(i = 0; i < ds->num_signals; i++)
			if(ds->signal[i])
				printf(" (%d,%f)", ds->signal[i]->id, ds->signal[i]->val);
	}
	if(ds->num_antigen) {
		for(i = 0; i < ds->num_antigen; i++)
			if(ds->antigen[i]) {
				printf(" (%d,", ds->antigen[i]->id);
				fwrite((void *) ds->antigen[i]->val, sizeof(char), \
						ds->antigen[i]->len, stdout);
				printf(")");
			}
	}
	printf("\n");
	fflush(stdout);
}

inline DS *pkt2ds(char *pkt, int len)
{
	DS *ds;
	register char *ptr = pkt;
	register int count = len, i;

	/* skip packet type */
	ptr++;

	/* copy structure */
	if(count < sizeof(DS))
		return NULL;
	ds = init_ds(0, 0);
	memcpy(ds, ptr, sizeof(DS));
	count -= sizeof(DS);
	ptr += sizeof(DS);

	/* copy signals */
	if(ds->num_signals) {
		if(!(ds->signal = calloc(ds->num_signals, sizeof(Signal *))))
			return NULL;
		for(i = 0; i < ds->num_signals; i++) {
			if(count < sizeof(Signal))
				return NULL;
			if(!(ds->signal[i] = malloc(sizeof(Signal))))
				return NULL;
			memcpy(ds->signal[i], ptr, sizeof(Signal));
			ptr += sizeof(Signal);
			count -= sizeof(Signal);
		}
	}

	/* copy antigen */
	if(ds->num_antigen) {
		if(!(ds->antigen = calloc(ds->num_antigen, sizeof(Antigen *))))
			return NULL;
		for(i = 0; i < ds->num_antigen; i++) {
			if(count < sizeof(Antigen))
				return NULL;
			if(!(ds->antigen[i] = malloc(sizeof(Antigen))))
				return NULL;
			memcpy(ds->antigen[i], ptr, sizeof(Antigen));
			ptr += sizeof(Antigen);
			count -= sizeof(Antigen);
			if(ds->antigen[i]->len) {
				if(!(ds->antigen[i]->val = malloc(ds->antigen[i]->len * \
						sizeof(char))))
					return NULL;
				if(count < ds->antigen[i]->len)
					return NULL;
				memcpy(ds->antigen[i]->val, ptr, ds->antigen[i]->len);
				ptr += ds->antigen[i]->len;
				count -= ds->antigen[i]->len;
			}
		}
	}

	return ds;
}

inline int ds2pkt(DS *ds, char *pkt, int len)
{
	register char *ptr = pkt;
	register int count = len, i;

	/* set packet type */
	if(count < 1)
		return -1;
	*ptr++ = CLIENT_DS;
	count -= 1;

	/* copy datastream structure */
	if(count < sizeof(DS))
		return -1;
	memcpy(ptr, ds, sizeof(DS));
	ptr += sizeof(DS);
	count -= sizeof(DS);

	/* copy signals */
	if(ds->num_signals)
		for(i = 0; i < ds->num_signals; i++) {
			if(count < sizeof(Signal))
				return -1;
			memcpy(ptr, ds->signal[i], sizeof(Signal));
			ptr += sizeof(Signal);
			count -= sizeof(Signal);
		}

	/* copy antigen */
	if(ds->num_antigen)
		for(i = 0; i < ds->num_antigen; i++) {
			if(count < sizeof(Antigen))
				return -1;
			memcpy(ptr, ds->antigen[i], sizeof(Antigen));
			ptr += sizeof(Antigen);
			count -= sizeof(Antigen);
			if(count < ds->antigen[i]->len)
				return -1;
			memcpy(ptr, ds->antigen[i]->val, ds->antigen[i]->len);
			ptr += ds->antigen[i]->len;
			count -= ds->antigen[i]->len;
		}

	return len - count;
}

inline void log_ds(DS *ds)
{
	static char message[1024];
	register char *ptr;
	register unsigned int i, len;

	ptr = message;
	len = sprintf(ptr, "ds %d.%06d %d %d %d", \
			(int) ds->tm.tv_sec, (int) ds->tm.tv_usec, \
			ds->id, ds->num_signals, ds->num_antigen);
	ptr += len;
	if(ds->num_signals) {
		for(i = 0; i < ds->num_signals; i++)
			if(ds->signal[i]) {
				len = sprintf(ptr, " (%d,%f)", \
						ds->signal[i]->id, ds->signal[i]->val);
				ptr += len;
			}
	}
	if(ds->num_antigen) {
		for(i = 0; i < ds->num_antigen; i++)
			if(ds->antigen[i]) {
				len = sprintf(ptr, " (%d,", ds->antigen[i]->id);
				ptr += len;
				memcpy(ptr, (void *) ds->antigen[i]->val, ds->antigen[i]->len);
				ptr += ds->antigen[i]->len;
				len = sprintf(ptr, ")");
				ptr += len;
			}
	}
	*ptr = '\0';

	logtime_tissue(message);
}
