/******************************************************************************/
/* (c) 2006 jamie twycross, jpt AT cs.nott.ac.uk                              */
/* released under gnu gpl v2                                                  */
/* tissue api                                                                 */
/******************************************************************************/

#ifndef TISSUE_H
#define TISSUE_H

#include "misc.h"
#include "log.h"
#include "antigen.h"
#include "cytokine.h"
#include "cell.h"
#include "ds.h"
#include "client.h"
#include "response.h"

#include <pthread.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#define TS_DEFAULT_PORT		5050  /* default port tissue server listens on */
#define TS_BUFFER_SIZE		10000  /* server input buffer size */

/* types for tissue cytokines */
#define CYTOKINE_REFLECTIVE		0  /* value replaces current tissue value */
#define CYTOKINE_ADDITIVE		1  /* value is added to current tissue value */
#define CYTOKINE_CUSTOM			2  /* tissue value function of current, value */

#define DEF_DECAY_RATE 		1.0  /* rate at which cytokines decay in tissue */
#define DEF_CYTOKINE_TYPE	CYTOKINE_REFLECTIVE  /* tissue cytokine type */

#define CLIENT_MSG		0x00
#define CLIENT_DS		0x01

/* tissue composed of antigen, cytokines + cells */
typedef struct Tissue {
	pthread_mutex_t *lock;
	pthread_t update_thread;
	pthread_t server_thread;

	unsigned int num_antigen;
	Antigen **antigen;
	unsigned int num_cytokines;
	Cytokine **cytokine;
	int *cytokine_type;
	unsigned int num_cells;
	Cell **cell;
	double cytokine_decay_rate;

	unsigned int antigen_multiplier;
	
	/* tissue server */
	int sd;
	struct sockaddr_in sa;
	short int log;

	/* logging flags */
	short int log_antigen;  /* log tissue antigen */
	short int log_cytokines;  /* log tissue cytokine */
	short int log_cells;  /* log tissue cells */
	short int log_tissue;  /* log all 3 of above */

	/* probe */
	pthread_t probe_thread;
	void (*probe)(struct Tissue *);
	unsigned long probe_rate;

	/* tcreplay synchronization - custom addition */
	long signalts;  /* Handling cases where server is faster */
	

} Tissue;

/* function prototypes */
Tissue *init_tissue(const int16_t, const unsigned int, const unsigned int, \
		const unsigned int);
void free_tissue(Tissue *);
void print_tissue(Tissue *);
void print_tissue_cytokines(Tissue *);
void print_tissue_antigen(Tissue *);
void print_tissue_cells(Tissue *);
void *handle_tissue_server(void *);
void *handle_client(void *);
void set_cell_input(Tissue *, const unsigned int);
void set_tissue(Tissue *, const unsigned int);
void step_tissue(Tissue *);
int read_parameter_uint(char *, char *, unsigned int *);
void log_parameter_file(char *);
void *handle_probe(void *);
void init_probe(Tissue *, void (*)(), unsigned long);

#endif
