/******************************************************************************/
/* dcak                                                                       */
/*                                                                            */
/*                                                                            */
/******************************************************************************/

#include <tissuek/tissue.h>
#include <tissuek/syscall.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <err.h>
#include <signal.h>
#include <math.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <asm/unistd.h>


/* model parameters */
unsigned int max_antigen=500;
unsigned int max_cytokines=6;
unsigned int max_cells=200;
unsigned int cell_update_rate=1000000;
unsigned int antigen_multiplier=10;
unsigned int probe_rate=1000000;
unsigned int num_cells_1=100;
unsigned int cell_lifespan_1=40;
unsigned int num_antigen_1=50;
unsigned int num_cytokines_1=3;
unsigned int num_antigen_receptors_1=1;
unsigned int num_cytokine_receptors_1=3;
unsigned int cell_lifespan_2=1;
unsigned int num_antigen_2=50;
unsigned int num_cytokines_2=3;
unsigned int num_response_producers_2=1;
unsigned int cell_lifespan_3=1;
unsigned int num_antigen_3=50;
unsigned int num_cytokines_3=3;
unsigned int num_response_producers_3=1;
float signal_scale_factor_1=1.0;
float signal_scale_factor_2=1.0;
float signal_scale_factor_3=1.0;
float max_sig_value1=100.0;
float max_sig_value2=100.0;
float max_sig_value3=10.0;
unsigned int log_tissue=1;
unsigned int ts_port = TS_DEFAULT_PORT;


char parameter_file[1024] = "";
char results_file[1024] = "results.log";
char response_file[1024] = "response.log";

FILE *fpresultsfile, *fpresponsefile;

unsigned int uniqueid = 1;


Tissue *tissue;


void mature_dc(Cell *cell);
Cell *create_idc(void);



/* probe compartment periodically + save data to log */
void probe_tissue(Tissue *tissue)
{
        register unsigned int i, j;
        static char message[32768], *ptr;

        ptr = message;
        ptr += sprintf(ptr, "tissue");
        for(i = 0; i < tissue->num_cytokines; i++)
                ptr += sprintf(ptr, " %f", tissue->cytokine[i]->val);
        logtime_tissue(message);

        for(i = 0; i < tissue->num_cells; i++)
	{
                if(tissue->cell[i]) {
                        if(tissue->cell[i]->type == 1) {  /* iDC */
                                ptr = message;
                                ptr += sprintf(ptr, "cell 1 %d %d %f %f %f", \
                                                tissue->cell[i]->id, \
                                                tissue->cell[i]->iterations, \
						tissue->cell[i]->cytokine[0]->val, \
						tissue->cell[i]->cytokine[1]->val, \
						tissue->cell[i]->cytokine[2]->val ); 
                                for(j = 0; j < num_cytokine_receptors_1; j++)
                                        ptr += sprintf(ptr, " %f", \
                                                        tissue->cell[i]->cytokine_receptor[j]->val);
                                for(j = 0; j < tissue->cell[i]->num_antigen; j++)
                                        if(tissue->cell[i]->antigen[j])
                                                ptr += sprintf(ptr, " %s", \
                                                                antigentostr(tissue->cell[i]->antigen[j]));
                                logtime_tissue(message);
                                continue;
                        }

		/* STILL TO DO - probe for cell type 2 and 3 */
                        if(tissue->cell[i]->type == 2 || tissue->cell[i]->type == 3) {  /* smDC or mDC */
                                ptr = message;
                                ptr += sprintf(ptr, "cell %d %d %d %f %f %f", \
                                                tissue->cell[i]->type, \
                                                tissue->cell[i]->id, \
                                                tissue->cell[i]->iterations, \
						tissue->cell[i]->cytokine[0]->val, \
						tissue->cell[i]->cytokine[1]->val, \
						tissue->cell[i]->cytokine[2]->val ); 
                                for(j = 0; j < tissue->cell[i]->num_antigen; j++)
                                        if(tissue->cell[i]->antigen[j])
                                                ptr += sprintf(ptr, " %s", \
                                                                antigentostr(tissue->cell[i]->antigen[j]));
                                logtime_tissue(message);
                        }


		}


	}
}



/* produce a response */
char *response_message(Cell *cell)
{
        unsigned int i;
        char *message, *ptr;
		

	if(!(message = malloc(32768)))
	{
		err(1, "Couldn't allocate memory for response message");
	}

	ptr = message;

	/* print cell id and i cytokine */
	ptr += sprintf(ptr, "cell: %d\n", cell->id);

	/* loop cell antigen and associate with cell's k value */
	for(i = 0; i < num_antigen_2; i++)
	{
		if(cell->antigen[i])
		{
			ptr += sprintf(ptr, "antigen: %s - %d\n", antigentostr(cell->antigen[i]), cell->type);
			/*printf("%s - %lf - %lf\n",antigentostr(cell->antigen[i]),  cell->cytokine[1]->val, cell->cytokine[2]->val);*/
		}
	}

        return message;
}



/* find empty slot in tissue for cell */
unsigned int empty_cell_slot(Tissue *tissue)
{
        register unsigned int ind;

        for(ind = 0; ind < tissue->num_cells; ind++)
                if(!tissue->cell[ind])
                        return ind;
        /* fail if tissue full */
        logfatal_tissue("error: no cell slots left\n");
        return ind;
}




/* iDC cell cycle */
void cell_cycle_1(Cell *cell)
{

	unsigned int j, ind; 
	short int f_antigen = 0;

	/* Updating cell cytokine matrix */
	/* o_csm */
	cell->cytokine[0]->val -= tissue->cytokine[3]->val;

	/* o_m */
	cell->cytokine[1]->val += tissue->cytokine[4]->val;

	/* o_sm */
	cell->cytokine[2]->val += tissue->cytokine[5]->val;

	/* Maturation threshold reached */
	if (cell->cytokine[0]->val <= 0)
	{

       		/* only mature if collected antigen */
       		for(j = 0; j < cell->num_antigen; j++)
		{
       			if(cell->antigen[j])
			{
				f_antigen = 1;  
                		break;
			}
		}

		/* Mature DC and replace */
		if(f_antigen)
		{
			mature_dc(cell);
                	/* find spare slot in tissue to put cell */
                	ind = empty_cell_slot(tissue);
                	/* initialise the cell */
                	tissue->cell[ind] = create_idc(); 
		}


		/* Kill Cell (In case no antigen collected) and replace */
		if(!f_antigen)
		{
			cell->id = uniqueid++;
			cell->iterations = 0;
			cell->cytokine[0]->val = cell_lifespan_1; /* o_csm */	
			cell->cytokine[1]->val = 0; /* o_m */	
			cell->cytokine[2]->val = 0; /* o_sm */	
		}

	}

}



/* matDC cell cycle */
void cell_cycle_23(Cell *cell)
{

	/* DO NOTHING */
}



/* maturing an iDC into a semi/mature DC - cell type 2 and 3 - note code structure to accommodate extensibility */
void mature_dc(Cell *cell)
{
	unsigned int i;

	/*printf("Iterations: %d\n", cell->iterations);*/

	/* smDC */
	if(cell->cytokine[1]->val < cell->cytokine[2]->val ) 
	{ 
		cell->type = 2; 
		cell->num_antigen = num_antigen_2; 
		cell->num_cytokines = num_cytokines_2; 
		cell->num_cytokine_receptors = 0; 
		for(i = 0; i < num_cytokine_receptors_1; i++) 
		{ 
			free_cytokine_receptor(cell->cytokine_receptor[i]);	
		} 
		cell->num_antigen_receptors = 0;
		for(i = 0; i < num_antigen_receptors_1; i++) 
		{
			free_antigen_receptor(cell->antigen_receptor[i]); 	
		}
		cell->cycle = cell_cycle_23;
		cell->iterations = 0;
		cell->num_response_producers = 1;
		/* create and configure response producer */
               	if(!(cell->response_producer = calloc(1, \
                                sizeof(Response_Producer *))))
                        	err(1, "couldn't allocate memory for response producers");
		cell->response_producer[0] = init_response_producer();
		for(i = 0; i < num_antigen_2; i++)
		{
			if(cell->antigen[i])
			{
				set_rp_antigen(cell->response_producer[0], cell->antigen[i]);
				break;
			}
		}
		set_rp_response(cell->response_producer[0], response_message(cell));
	} else { /* mDC */
		cell->type = 3; 
		cell->num_antigen = num_antigen_3; 
		cell->num_cytokines = num_cytokines_3; 
		cell->num_cytokine_receptors = 0; 
		for(i = 0; i < num_cytokine_receptors_1; i++) 
		{ 
			free_cytokine_receptor(cell->cytokine_receptor[i]);	
		} 
		cell->num_antigen_receptors = 0;
		for(i = 0; i < num_antigen_receptors_1; i++) 
		{
			free_antigen_receptor(cell->antigen_receptor[i]); 	
		}
		cell->cycle = cell_cycle_23;
		cell->iterations = 0;
		cell->num_response_producers = 1;
		/* create and configure response producer */
               	if(!(cell->response_producer = calloc(1, \
                                sizeof(Response_Producer *))))
                        	err(1, "couldn't allocate memory for response producers");
		cell->response_producer[0] = init_response_producer();
		for(i = 0; i < num_antigen_3; i++)
		{
			if(cell->antigen[i])
			{
				set_rp_antigen(cell->response_producer[0], cell->antigen[i]);
				break;
			}
		}
		set_rp_response(cell->response_producer[0], response_message(cell));

	}



}


/* create iDC */
Cell *create_idc(void)
{
	Cell_Param cell_param;
        Cell *cell;

        /* set parameters for cell creation */
        bzero(&cell_param, sizeof(Cell_Param));
        cell_param.type = 1;  /* type 1 - iDC */
        cell_param.num_antigen = num_antigen_1; /* num of cell antigen */
        cell_param.num_cytokines = num_cytokines_1; /* num of cell cytokine*/
        cell_param.num_cytokine_receptors = num_cytokine_receptors_1;/* signal */
        cell_param.num_antigen_receptors = num_antigen_receptors_1;
        cell_param.cycle = cell_cycle_1;

        /* initialise the cell */
        cell = init_cell(&cell_param);

        /* set cell ID */
        cell->id = uniqueid++;

        /* set cytokine receptor to respond to tissue signal without any noise - note updated tissue.c */
	/* tissue cytokine - icmprate, outrate, inverseoutratechange, o_csm, o_m, o_sm */
        cell->cytokine_receptor[0]->id = 3;
        cell->cytokine_receptor[0]->noise = 0.0;
        cell->cytokine_receptor[1]->id = 4;
        cell->cytokine_receptor[1]->noise = 0.0;
        cell->cytokine_receptor[1]->id = 5;
        cell->cytokine_receptor[1]->noise = 0.0;

        /* set tissue cytokine values */
	cell->cytokine[0]->val = ( (double) (cell_lifespan_1/2)) +  ( rand() % cell_lifespan_1); /* o_csm */	
	cell->cytokine[1]->val = 0; /* o_m */	
	cell->cytokine[2]->val = 0; /* o_sm */	

        return cell;
}



/* initialise tissue cells */
void create_tissue_cells()
{
        register unsigned int i, ind;

        i = num_cells_1;
        while(i--) {
                /* find spare slot in tissue to put cell */
                ind = empty_cell_slot(tissue);
                /* initialise the cell */
                tissue->cell[ind] = create_idc();

                /* set the maturation threshold to a random value in order to avoid cell maturation lockstep - TAKEN OVER */
          	/*tissue->cell[ind]->cytokine[0]->val = (rand() % cell_lifespan_1)+1;*/
        }

}


/* Fusion Function ref: Greensmith et al. - DCA Articulation */
/* Will work properly only if the cell cycle is set to be equivalent to the signal input rate */
void fsig()
{

        pthread_mutex_lock(tissue->lock);

	tissue->cytokine[3]->val = (  ((( (double) 2) * tissue->cytokine[0]->val) * signal_scale_factor_1) +  \
		((( (double) 1) * tissue->cytokine[1]->val) * signal_scale_factor_2) + \
		( ((double) 2) * (max_sig_value3-(tissue->cytokine[2]->val * signal_scale_factor_3)) ) ); /* tissue o_csm */


	tissue->cytokine[4]->val = (  ((( (double) 2) * tissue->cytokine[0]->val) * signal_scale_factor_1) +  \
		((( (double) 1) * tissue->cytokine[1]->val) * signal_scale_factor_2) + \
		( ((double) -2) * (max_sig_value3-(tissue->cytokine[2]->val * signal_scale_factor_3)) ) ); /* tissue o_m */


	tissue->cytokine[5]->val = (  (((0) * tissue->cytokine[0]->val) * signal_scale_factor_1) +  \
		((( (double) 0) * tissue->cytokine[1]->val) * signal_scale_factor_2) + \
		( ((double) 2) * (max_sig_value3-(tissue->cytokine[2]->val * signal_scale_factor_3)) ) ); /* tissue o_sm */

        pthread_mutex_unlock(tissue->lock);

}


/* Stabilizing cell population  */
void stabilize_cell_population()
{

	unsigned int i;

        pthread_mutex_lock(tissue->lock);

	/* Open response file for writing */
        if(!(fpresponsefile = fopen(response_file, "a")))
                err(1, "error: couldn't open response file");
 
	/* looping all cells - searching for smDC and mDC's */
        for(i = 0; i < max_cells; i++) {
               if(tissue->cell[i]) {
               		if(tissue->cell[i]->type == 2 || tissue->cell[i]->type == 3) {  /* smDC or mDC */
				/* log response to response file */
				fprintf(fpresponsefile,"%s", tissue->cell[i]->response_producer[0]->response);

                               	/* kill (apoptosis) */
                                free_cell(tissue->cell[i]);
                                tissue->cell[i] = NULL;
                        }
		}
	}


	/* close responsefile */
	fclose(fpresponsefile);

        pthread_mutex_unlock(tissue->lock);
}


/* Calculate MCAV 's */
void calc_mcav()
{
	unsigned int i;
	char filename[64], line[1024]; 	
	char **uniqueantigen, antigenval[512] = "";
	int f_isantigeninmatrix = 0, numantigen, celltype; 
	double total2, total3, mcav;


        pthread_mutex_lock(tissue->lock); /* Tissue locking is required in case calc_mcav is called periodically */

	if(!(uniqueantigen = calloc(1000, sizeof(char*))))
	{
		err(1, "Couldn't allocate memory for unique antigen matrix\n");
	}


	/* appending filename with current time */
	time_t mytime = time(0);
	sprintf(filename, "%s_%s", results_file, asctime(localtime(&mytime))); 
	

	/* Open results file for writing */
        if(!(fpresultsfile = fopen(filename, "w")))
                err(1, "error: couldn't open results file\n");

	/* mock content for stub purposes */
	/* fprintf(fpresultsfile, "Stub content\n"); */


	/* identifying unique antigen and load into memory */
        if(!(fpresponsefile = fopen(response_file, "r")))
                err(1, "error: couldn't open response file\n");

	while(fgets(line, sizeof(line), fpresponsefile))
	{

		/* Case cell entry - skip */
		if(strstr(line,"cell:"))
		{
			continue;
		} else {/* Case antigen entry - check whether to include in uniqueantigen matrix */
			/* extract antigen from input line */
			if(sscanf(line, "antigen: %s - %d", antigenval, &celltype) == 0 )
			{ 
			 	err(1, "Response file format problem\n");
			}

			/* check whether antigen is already in matrix */
			f_isantigeninmatrix = 0;
			for(i = 0; i < 1000; i++)
			{
				if(uniqueantigen[i])
				{
					if(strcmp(antigenval,uniqueantigen[i]) == 0)
					{
						f_isantigeninmatrix = 1;
						break;
					}
				} else {
					break; /* only empty matrix slots beyond this point */
				} 

			}
			/* if not present, include it at the first empty slot in matrix */
			if(!f_isantigeninmatrix)
			{
				for(i = 0; i < 1000; i++)
				{
					if(!uniqueantigen[i])
					{
						if(!(uniqueantigen[i] = malloc(512)))
						{
							err(1, "Couldn't allocate memory for unique antigen matrix entry\n");
						}
						strcpy(uniqueantigen[i], antigenval);
						break;
					}

				}
				if(i == 1000)
					err(1, "Ran out of unqiueantigen matrix memory space \n");
			}

		}

	}

	fclose(fpresponsefile);


	/* Test uniqueantigen matrix */
	for(i = 0; i < 1000; i++)
	{
		printf("Antigen matrix value at position %d is %s\n",i, uniqueantigen[i]);
	}


	/* For each unique antigen - calculate MCAV and write to results file */
	for(i = 0; i < 1000; i++)
	{
		if(uniqueantigen[i])
		{
			/* calculating MVAC for current antigen in unique antigen matrix */
			numantigen = 0;
			total2 = 0; total3 =0;
        		if(!(fpresponsefile = fopen(response_file, "r")))
		                err(1, "error: couldn't open response file\n");
			while(fgets(line, sizeof(line), fpresponsefile))
			{
				/* Case cell entry - skip */
				if(strstr(line,"cell:"))
				{
					continue;
				} else {/* Case antigen entry - check whether it is the antigen currently being processed */
				/* extract antigen from input line */
					if(sscanf(line, "antigen: %s - %d", antigenval, &celltype) == 0 )
					{ 
			 			err(1, "Response file format problem\n");
					}
				
					/* our antigen */
					if(strcmp(uniqueantigen[i], antigenval) == 0)
					{
						numantigen++;
						if(celltype == 2)
						{
							total2++;
						} else {
							total3++;
						}
					}

				}

			}

			/* calculate mcav for current antigen and write to file  */
			mcav = total3 / numantigen ;  /* MCAV for antigen */
			fprintf(fpresultsfile, "%s - %lf\n", uniqueantigen[i], mcav);
			fclose(fpresponsefile);

		} else {
			break; /* only empty matrix slots from this point onwards */
		}
	}



	/* Re-setting cumulative tissue cytokines - for when calc_mcav is called periodically  */
	/* none in this case */

	/* close resultsfile */
	fclose(fpresultsfile);

	/* free uniqueantigen matrix memory */
	for(i = 0; i < 1000; i++)
	{
		free(uniqueantigen[i]);
	}
	free(uniqueantigen);


        pthread_mutex_unlock(tissue->lock); /* Tissue locking is required in case calc_mcav is call periodically */

}



/* Handle Ctrl+C SIGINT signal */
void handle_shutdown()
{

	printf("Handling shutdown\n");

	/* calculating MCAV for antigen responded to during execution */
	calc_mcav();

	/* clean up  - calling original tissue server shutdown handling procedure */
	free_tissue(tissue);

	/* exiting with success */
	exit(EXIT_SUCCESS);

}



int read_parameter_float(char *filename, char *parameter, float *val)
{
        FILE *fp;
        char line[1024], *ptr, *endptr[1];

        if(!(fp = fopen(filename, "r")))
                err(1, "error: couldn't read parameter file");

        while(fgets(line, sizeof(line), fp)) {
                if(line[0] == '#')  /* comment line */
                        continue;
                if(!(ptr = index(line, '=')))
                        err(1, "error: couldn't parse parameter file");
                *ptr = '\0';
                if(strcmp(line, parameter) == 0) {
                        /* it's our parameter */
                        *val = (float) strtod(ptr + 1, endptr);
                        if(**endptr != '\n')
                                err(1, "error: couldn't parse parameter file");
                        fclose(fp);
                        return 0;  /* found */
                }
        }

        fclose(fp);
        return 1;  /* not found */
}


/* read parameters from file */
void read_parameters(char *filename)
{
        read_parameter_uint(filename, "max_antigen", &max_antigen);
        read_parameter_uint(filename, "max_cytokines", &max_cytokines);
        read_parameter_uint(filename, "max_cells", &max_cells);
        read_parameter_uint(filename, "cell_update_rate", &cell_update_rate);
        read_parameter_uint(filename, "antigen_multiplier", &antigen_multiplier);
        read_parameter_uint(filename, "probe_rate", &probe_rate);
        read_parameter_uint(filename, "num_cells_1", &num_cells_1);
        read_parameter_uint(filename, "cell_lifespan_1", &cell_lifespan_1);
        read_parameter_uint(filename, "num_antigen_1", &num_antigen_1);
        read_parameter_uint(filename, "num_cytokines_1", &num_cytokines_1);
        read_parameter_uint(filename, "num_antigen_receptors_1", \
                        &num_antigen_receptors_1);
        read_parameter_uint(filename, "num_cytokine_receptors_1", \
                        &num_cytokine_receptors_1);
        read_parameter_uint(filename, "cell_lifespan_2", &cell_lifespan_2);
        read_parameter_uint(filename, "num_antigen_2", &num_antigen_2);
        read_parameter_uint(filename, "num_cytokines_2", &num_cytokines_2);
        read_parameter_uint(filename, "num_response_producers_2", &num_response_producers_2);
        read_parameter_uint(filename, "cell_lifespan_3", &cell_lifespan_3);
        read_parameter_uint(filename, "num_antigen_3", &num_antigen_3);
        read_parameter_uint(filename, "num_cytokines_3", &num_cytokines_3);
        read_parameter_uint(filename, "num_response_producers_3", &num_response_producers_3);
        read_parameter_float(filename, "signal_scale_factor_1", &signal_scale_factor_1);
        read_parameter_float(filename, "signal_scale_factor_2", &signal_scale_factor_2);
        read_parameter_float(filename, "signal_scale_factor_3", &signal_scale_factor_3);
        read_parameter_float(filename, "max_sig_value1", &max_sig_value1);
        read_parameter_float(filename, "max_sig_value2", &max_sig_value2);
        read_parameter_float(filename, "max_sig_value3", &max_sig_value3);
        read_parameter_uint(filename, "ts_port", &ts_port);
        read_parameter_uint(filename, "log_tissue", &log_tissue);
}


static void parse_command_line(int argc, char *argv[])
{
        if(argc == 2) {
                strncpy(parameter_file, argv[1], sizeof(parameter_file));
                read_parameters(parameter_file);
        } else {
                fprintf(stderr, "usage: dcak PARAMETER_FILE \n");
                exit(1);
        }
}


int main(int argc, char *argv[])
{
	struct sigaction siga;
	long currentsignalts = 0; /* tcreplay synchronization */

	printf("DCAM started ...\n");
	
	parse_command_line(argc, argv);

	/* create tissue compartment */
	tissue = init_tissue((int16_t) ts_port, max_antigen, max_cytokines, max_cells);

        /* save user specified parameters to log */
        log_parameter_file(parameter_file);

        /* log input data from tissue clients to compartment */
        tissue->log = (short int) log_tissue;

        /* set multiplier for incoming antigen */
        tissue->antigen_multiplier = antigen_multiplier;

	/* initialize cells in tissue compartment */
	create_tissue_cells();

        /* initialise a probe to monitor compartment */
        init_probe(tissue, probe_tissue, probe_rate);

	/* set sigaction to call calc_mcav as well as gracefully shutting down the tissue server */
	siga.sa_handler = handle_shutdown;
	sigemptyset(&siga.sa_mask);
	siga.sa_flags = SA_RESTART;
	if(sigaction(SIGINT, &siga, NULL) == -1)
	{
		err(1,"Couldn't set up sigint handler"); 
	}

	/* resetting responsefile.log from any possible previous sessions */
        if(!(fpresponsefile = fopen(response_file, "w"))) {
                err(1, "error resetting response file");
        }
	fclose(fpresponsefile);	


	/* run dcak algorithm */
	while(1)
	{
		 if(tissue->signalts != currentsignalts)
		 { 

			currentsignalts = tissue->signalts;
		
			fsig(); /* Note that after fsig() the thread client could make its way before step_tissue() does */
				/* still step_tissue reads from the fused D and S signals just computed */
			/* printf(".");*/
			step_tissue(tissue);	
			stabilize_cell_population();
			usleep(cell_update_rate);

		 } 
	}

	 return 0;
}



