/******************************************************************************/
/* (c) 2006 jamie twycross, jpt AT cs.nott.ac.uk                              */
/* released under gnu gpl v2                                                  */
/* antigen api                                                                */
/******************************************************************************/

#include "antigen.h"

#include <stdlib.h>
#include <err.h>
#include <string.h>
#include <ctype.h>

/* initialise an antigen */
Antigen *init_antigen(const unsigned int len)
{
	Antigen *antigen;

	if(!(antigen = malloc(sizeof(Antigen))))
		err(1, "couldn't allocate memory for antigen");
	antigen->id = 0;
	antigen->len = len;
	if(antigen->len)
		if(!(antigen->val = malloc(antigen->len * sizeof(char))))
			err(1, "couldn't allocate memory for antigens");

	return antigen;
}

/* free memory of an antigen */
void free_antigen(Antigen *antigen)
{
	if(!antigen)
		return;
	if(antigen->len)
		free(antigen->val);
	free(antigen);
}

/* read antigen from stream */
Antigen *read_antigen(FILE *fp)
{
	Antigen *antigen;
	int id;
	unsigned int len;

	if(fread((void *) &id, 1, sizeof(int), fp) != sizeof(int))
		err(1, "couldn't read antigen->id");
	if(fread((void *) &len, 1, sizeof(unsigned int), fp) != \
			sizeof(unsigned int))
		err(1, "couldn't read antigen->len");
	antigen = init_antigen(len);
	antigen->id = id;
	if(fread((void *) antigen->val, sizeof(char), antigen->len, fp) != \
			antigen->len)
		err(1, "couldn't read antigen->val");

	return antigen;
}

/* write antigen to stream */
void write_antigen(Antigen *antigen, FILE *fp)
{
	if(fwrite((void *) &antigen->id, 1, sizeof(int), fp) != sizeof(int))
		err(1, "couldn't write antigen->id");
	if(fwrite((void *) &antigen->len, 1, sizeof(unsigned int), fp) != \
			sizeof(unsigned int))
		err(1, "couldn't write antigen->len");
	if(fwrite((void *) antigen->val, sizeof(char), antigen->len, fp) != \
			antigen->len)
		err(1, "couldn't write antigen->val");
}

/* create a duplicate copy of an antigen (allocates memory) */
inline Antigen *dup_antigen(Antigen *antigen)
{
	Antigen *copy;

	copy = init_antigen(antigen->len);
	copy->id = antigen->id;
	memcpy(copy->val, antigen->val, sizeof(char) * antigen->len);

	return copy;
}

/* sets the value of an antigen */
/* antigen must initialised beforehand, but can be of different lens */
inline Antigen *set_antigen(Antigen *antigen, char *val)
{
	register unsigned int len;

	len = strlen(val);
	if(!antigen)
		antigen = init_antigen(len);
	if(len != antigen->len) {
		free(antigen->val);
		if(!(antigen->val = malloc(len * sizeof(char))))
			err(1, "couldn't allocate memory for antigens");
		antigen->len = len;
	}
	memcpy(antigen->val, val, sizeof(char) * len);

	return antigen;
}

/* sets the value of an antigen */
/* antigen must initialised beforehand + of same len */
inline void set_antigen_fast(Antigen *antigen, char *val)
{
	memcpy(antigen->val, val, sizeof(char) * antigen->len);
}

/* simple antigen matching */
inline double antigen_match_simple(Antigen *lock, Antigen *key)
{
	register unsigned int lockind, keyind, count;

	/* count number of places lock matches key */
	/* value 0 in lock is a wildcard */
	if(!key || !lock)
		return 0.0;
	keyind = lockind = count = 0;
	while(lockind < lock->len) {
		if(keyind >= key->len)
			break;  /* lock bigger than key */
		if(!lock->val[lockind])  /* wildcard */
			count++;
		else
			if(lock->val[lockind] == key->val[keyind])  /* matching value */
				count++;
		keyind++;
		lockind++;
	}

	return (double) count / lock->len;
}

/* syscall antigen matching */
inline double antigen_match_syscall(Antigen *lock, Antigen *key)
{
	if(!key || !lock)
		return 0.0;
	if(antigentoi(lock) != antigentoi(key))
		return 0.0;
	return 1.0;
}

/* string antigen matching */
/* lock + key must be of same length */
inline double antigen_match_string(Antigen *lock, Antigen *key)
{
	if(!key || !lock)
		return 0.0;
	if(memcmp(lock->val, key->val, lock->len) != 0)
		return 0.0;
	return 1.0;
}

inline int antigentoi(Antigen *antigen)
{
	register unsigned int i, val;
	register char c;

	if(!antigen || !antigen->len)
		return 0;
	i = val = 0;
	while(i < antigen->len) {
		if(!isdigit(c = antigen->val[i++]))
			break;
		val = val * 10 + (c - '0');
	}

	return val;
}

inline long antigentolong(Antigen *antigen)
{
	register long i, val;
	register char c;

	if(!antigen || !antigen->len)
		return 0;
	i = val = 0;
	while(i < antigen->len) {
		if(!isdigit(c = antigen->val[i++]))
			break;
		val = val * 10 + (c - '0');
	}

	return val;
}

/* !! not thread-safe */
inline const char *antigentostr(Antigen *antigen)
{
	static char str[64];

	memcpy(str, antigen->val, antigen->len); /* !! */
	str[antigen->len] = '\0';
	
	return str;
}
