/******************************************************************************/
/* (c) 2006 jamie twycross, jpt AT cs.nott.ac.uk                              */
/* released under gnu gpl v2                                                  */
/* tissue api                                                                 */
/******************************************************************************/

#include "tissue.h"

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <time.h>
#include <arpa/inet.h>
#include <netinet/sctp.h>
#include <netdb.h>
#include <strings.h>
#include <string.h>
#include <err.h>

#define BACKLOG 100

static Tissue *tcopy = NULL;  /* copy of tissue for sigint handler*/

static void sigint_handler(int s)
{
	free_tissue(tcopy);
	exit(EXIT_SUCCESS);
}

/* initialise the tissue */
Tissue *init_tissue(const int16_t server_port, const unsigned int num_antigen, \
		const unsigned int num_cytokines, const unsigned int num_cells)
{
	Tissue *tissue;
	unsigned int i;
	struct sigaction siga;

	srand(time(NULL));
	srand48(time(NULL));

	/* setup logfile */
	open_logfile();
	logtime_tissue("starting tissue server");

	/* allocate memory */
	if(!(tissue = malloc(sizeof(Tissue))))
		logfatal_tissue("couldn't allocate memory for tissue");
	if(!(tissue->lock = malloc(sizeof(pthread_mutex_t))))
		logfatal_tissue("couldn't allocate memory for tissue mutex");
	tissue->num_antigen = num_antigen;
	if(tissue->num_antigen)
		if(!(tissue->antigen = calloc(tissue->num_antigen, sizeof(Antigen *))))
			logfatal_tissue("couldn't allocate memory for antigen");
	tissue->num_cytokines = num_cytokines;
	if(tissue->num_cytokines) {
		if(!(tissue->cytokine = calloc(tissue->num_cytokines, \
				sizeof(Cytokine *))))
			logfatal_tissue("couldn't allocate memory for cytokines");
		if(!(tissue->cytokine_type = calloc(tissue->num_cytokines, \
				sizeof(int))))
			logfatal_tissue("couldn't allocate memory for cytokine types");
		for(i = 0; i < tissue->num_cytokines; i++) {
			tissue->cytokine[i] = init_cytokine();
			tissue->cytokine[i]->id = i;
			tissue->cytokine_type[i] = DEF_CYTOKINE_TYPE;
		}
	}
	tissue->num_cells = num_cells;
	if(tissue->num_cells)
		if(!(tissue->cell = calloc(tissue->num_cells, sizeof(Cell *))))
			logfatal_tissue("couldn't allocate memory for cells");

	/* capture INT to exit cleanly */
	tcopy = tissue;
	siga.sa_handler = sigint_handler;
	sigemptyset(&siga.sa_mask);
	siga.sa_flags = SA_RESTART;
	if(sigaction(SIGINT, &siga, NULL) == -1)
		logfatal_tissue("couldn't setup sigint handler");

	/* set defaults */
	tissue->log = 0;  /* no logging of server */
	tissue->cytokine_decay_rate = DEF_DECAY_RATE;  /* cytokine decay rate */
	tissue->probe = NULL;
	tissue->antigen_multiplier = 1;  /* how many copies of incoming antigen */
	tissue->signalts = 0;

	/* start threads to handle tissue server */
	pthread_mutex_init(tissue->lock, NULL);
	if(pthread_create(&tissue->server_thread, NULL, handle_tissue_server, \
			(void *) tissue))
		logfatal_tissue("couldn't create tissue server thread");

	return tissue;
}

/* free memory for tissue structure */
void free_tissue(Tissue *tissue)
{
	logtime_tissue("stopping tissue server");

	if(!tissue)
		return;

	/* kill the update thread */
	if(tissue->probe)
		pthread_cancel(tissue->probe_thread);
	pthread_cancel(tissue->server_thread);
	pthread_mutex_destroy(tissue->lock);

	/* close server */
	close(tissue->sd);

	/* close log */
	close_logfile();

	/* and clean up memory... */
}

void print_tissue(Tissue *tissue)
{
	unsigned int i;

	pthread_mutex_lock(tissue->lock);
	printf("tissue_cytokines=%d: ", tissue->num_cytokines);
	for(i = 0; i < tissue->num_cytokines; i++)
		printf(" (%d,%f)", tissue->cytokine[i]->id, tissue->cytokine[i]->val);
	printf("\ntissue_antigen=%d: ", tissue->num_antigen);
	for(i = 0; i < tissue->num_antigen; i++)
		if(tissue->antigen[i]) {
			printf(" (%d,", tissue->antigen[i]->id);
			fwrite((void *) tissue->antigen[i]->val, sizeof(char), \
					tissue->antigen[i]->len, stdout);
			printf(")");
		}
	for(i = 0; i < tissue->num_cells; i++)
		if(tissue->cell[i]) {
			printf("\ntissue_cell_%d: ", i);
			print_cell(tissue->cell[i]);
		}
	pthread_mutex_unlock(tissue->lock);
}

void print_tissue_cytokines(Tissue *tissue)
{
	unsigned int i;

	pthread_mutex_lock(tissue->lock);
	printf("tissue_cytokines:");
	for(i = 0; i < tissue->num_cytokines; i++)
		printf(" (%d,%f)", tissue->cytokine[i]->id, tissue->cytokine[i]->val);
	printf("\n");
	pthread_mutex_unlock(tissue->lock);
}

void print_tissue_antigen(Tissue *tissue)
{
	unsigned int i, count = 0;

	pthread_mutex_lock(tissue->lock);
	printf("tissue_antigen:");
	for(i = 0; i < tissue->num_antigen; i++)
		if(tissue->antigen[i]) {
			printf(" (%d,", tissue->antigen[i]->id);
			fwrite((void *) tissue->antigen[i]->val, sizeof(char), \
					tissue->antigen[i]->len, stdout);
			printf(")");
			count++;
		}
	printf(" total_antigen=%d\n", count);
	pthread_mutex_unlock(tissue->lock);
}

void print_tissue_cells(Tissue *tissue)
{
	unsigned int i, count = 0;

	pthread_mutex_lock(tissue->lock);
	for(i = 0; i < tissue->num_cells; i++)
		if(tissue->cell[i]) {
			printf("tissue_cell_%d: ", i);
			print_cell(tissue->cell[i]);
			count++;
		}
	printf("total_tissue_cells=%d\n", count);
	pthread_mutex_unlock(tissue->lock);
}

/* thread callback to handle input to the tissue server for tissue clients */
void *handle_tissue_server(void *arg)
{
	Tissue *tissue = (Tissue *) arg;
	int csd;
	struct sockaddr_in caddr;
	socklen_t clen;
	pthread_t client_thread;
	void *args[2];

	/* setup tissue server */
	memset(&tissue->sa, 0, sizeof(tissue->sa));
	tissue->sa.sin_family = AF_INET;
	tissue->sa.sin_addr.s_addr = htonl(INADDR_ANY);
	tissue->sa.sin_port = htons(TS_DEFAULT_PORT);
	if((tissue->sd = socket(AF_INET, SOCK_STREAM, IPPROTO_SCTP)) == -1)
		logfatal_tissue("couldn't create tissue socket");
	if(bind(tissue->sd, (struct sockaddr *) &tissue->sa, sizeof(tissue->sa)) \
			== -1)
		logfatal_tissue("couldn't bind tissue socket");
	if(listen(tissue->sd, BACKLOG) == -1)
		logfatal_tissue("couldn't listen on socket");

	while(1) {
		/* wait for a connection */
		clen = sizeof(caddr);
		if((csd = accept(tissue->sd, (struct sockaddr *) &caddr, &clen)) == -1)
			logfatal_tissue("error: couldn't accept on socket\n");

		/* create thread to handle connection */
		args[0] = (void *) tissue;
		args[1] = (void *) &csd;
		if(pthread_create(&client_thread, NULL, handle_client, (void *) args))
			logfatal_tissue("couldn't create client thread");
	}
}

/* handle tissue client once connected */
void *handle_client(void *arg)
{
	void **args = (void **) arg;
	Tissue *tissue = (Tissue *) args[0];
	int csd = *((int *) args[1]);
	int len;
	static int type, ind;
	static char buffer[TS_BUFFER_SIZE];
	register unsigned int i, j;
	DS *ds;


	/* receive data */
	while((len = recv(csd, buffer, TS_BUFFER_SIZE, 0)) > 0) {
		/* process data */
		type = (int) *buffer;
		switch(type) {
			/* 0x01 - receive datastream event */
			case CLIENT_DS:
				if(!(ds = pkt2ds(buffer, len)))
					logfatal_tissue("couldn't parse datastream");
				/* set tissue from datastream */
				pthread_mutex_lock(tissue->lock);
				/* set cytokine levels */
				for(i = 0; i < ds->num_signals; i++) {

					tissue->signalts = (long) time(NULL);  /* Custom addition */

					if(ds->signal[i]->id >= 0 && ds->signal[i]->id < \
							tissue->num_cytokines)
						switch(tissue->cytokine_type[ds->signal[i]->id]) {
							/* replace current value in tissue with incoming */
							case CYTOKINE_REFLECTIVE:
								tissue->cytokine[ds->signal[i]->id]->val = \
										ds->signal[i]->val;
								break;

							case CYTOKINE_ADDITIVE:
							/* add incoming to current value in tissue */
								tissue->cytokine[ds->signal[i]->id]->val += \
										ds->signal[i]->val;
								break;

							/* new is a function of current and incoming */
							case CYTOKINE_CUSTOM:
								break;

							default:
								logtime_tissue("unknown cytokine type");
								break;
						}

				}
				/* set antigen in tissue */
				/* create multiple copies according to antigen_multiplier */
				for(j = 0; j < tissue->antigen_multiplier; j++)
					for(i = 0; i < ds->num_antigen; i++) {
						ind = (int) (((double) tissue->num_antigen * rand()) \
								/ (RAND_MAX + 1.0));  /* random insertion */
						if(tissue->antigen[ind])
							free_antigen(tissue->antigen[ind]);
						tissue->antigen[ind] = dup_antigen(ds->antigen[i]);
					}
				pthread_mutex_unlock(tissue->lock);
				if(tissue->log)
					log_ds(ds);

				/* tidy up */
				free_ds(ds);
				break;

			case CLIENT_MSG:
				logtime_tissue(buffer + 1);
				break;

			default:
				close(csd);
				logfatal_tissue("unknown message type");
				break;
		}
	}

	if(len == -1)
		logfatal_tissue("couldn't receive socket data");

	close(csd);
	pthread_exit(NULL);
}

/* set the input of a cell from the tissue */
void set_cell_input(Tissue *tissue, const unsigned int index)
{
	Cell *cell  = tissue->cell[index];
	unsigned int i, j, ind1, ind2;

	/* handle antigen receptors: transfer antigen from tissue to cell */
	for(i = 0; i < cell->num_antigen_receptors; i++) {
		/* get random antigen in tissue */
		ind1 = (int) (((double) tissue->num_antigen * rand()) / \
				(RAND_MAX + 1.0));
		/* only copy if antigen available */
		if(tissue->antigen[ind1]) {
			/* randomly copy to cell antigen store */
			ind2 = (int) (((double) cell->num_antigen * rand()) / \
					(RAND_MAX + 1.0));
			/* remove any previous antigen */
			free_antigen(cell->antigen[ind2]);
			/* copy antigen from cell to tissue */
			cell->antigen[ind2] = dup_antigen(tissue->antigen[ind1]);
			/* remove antigen from tissue */
			free_antigen(tissue->antigen[ind1]);
			tissue->antigen[ind1] = NULL;
		}
	}

	/* handle cytokine receptors: sample cytokine levels */
	for(i = 0; i < cell->num_cytokine_receptors; i++)
		/* only sample if cytokine available in tissue */
		if((cell->cytokine_receptor[i]->id < tissue->num_cytokines))
			cell->cytokine_receptor[i]->val = tissue->cytokine[cell->cytokine_receptor[i]->id]->val; /* Custom */

	/* handle cell receptors: look for other cells */
	for(i = 0; i < cell->num_cell_receptors; i++) {
		cell->cell_receptor[i]->cell = NULL;
		if(drand48() > cell->cell_receptor[i]->efficiency)
			continue;
		/* get random cell in tissue */
		ind1 = (int) (((double) (tissue->num_cells - 1) * rand()) / \
				(RAND_MAX + 1.0));
		if(ind1 >= index)  /* don't attach receptor to self */
			ind1++;
		if(!tissue->cell[ind1])
			continue;
		/* only conjunct with cell if same type and state as receptor likes */
		if((cell->cell_receptor[i]->type == -1) \
				|| cell->cell_receptor[i]->type == tissue->cell[ind1]->type)
			if((cell->cell_receptor[i]->state == -1) \
					|| cell->cell_receptor[i]->state == \
					tissue->cell[ind1]->state)
				cell->cell_receptor[i]->cell = tissue->cell[ind1];
	}

	/* handle vr receptors: match antigen */
	for(i = 0; i < cell->num_vr_receptors; i++) {
		/* can only examine antigen from cell conjoined via cell receptor */
		cell->vr_receptor[i]->cell = NULL;
		cell->vr_receptor[i]->activated = 0;
		free_antigen(cell->vr_receptor[i]->key);
		cell->vr_receptor[i]->key = NULL;
		randomise_list_2(cell->num_cell_receptors);
		for(j = 0; j < cell->num_cell_receptors; j++) {
			if(!cell->cell_receptor[randlist2[j]]->cell)
				continue;
			if((cell->vr_receptor[i]->type != -1) && \
					(cell->vr_receptor[i]->type != \
					cell->cell_receptor[randlist2[j]]->cell->type))
				continue;
			if((cell->cell_receptor[randlist2[j]]->state != -1) && \
					(cell->vr_receptor[i]->state != \
					cell->cell_receptor[randlist2[j]]->cell->state))
				continue;
			cell->vr_receptor[i]->cell = \
					cell->cell_receptor[randlist2[j]]->cell;
			break;  /* matching cell found */
		}
		if(!cell->vr_receptor[i]->cell)
			continue;  /* no matching cell found for receptor */
		if(!cell->vr_receptor[i]->cell->num_antigen_producers) {
			/* matching cell isn't producing antigen */
			cell->vr_receptor[i]->cell = NULL;
			continue;
		}
		/* see if antigen matches any of those on antigen producers */
		randomise_list_2(cell->vr_receptor[i]->cell->num_antigen_producers);
		for(j = 0; j < cell->vr_receptor[i]->cell->num_antigen_producers; j++)
			if(cell->vr_receptor[i]->cell->antigen_producer[randlist2[j]]->antigen)
				if(cell->vr_receptor[i]->match(cell->vr_receptor[i]->lock, \
						cell->vr_receptor[i]->cell->antigen_producer[randlist2[j]]->antigen) >= cell->vr_receptor[i]->threshold) {
					cell->vr_receptor[i]->activated = 1;
					cell->vr_receptor[i]->key = dup_antigen( \
					  cell->vr_receptor[i]->cell->antigen_producer[randlist2[j]]->antigen);
					break;
				}
	}

	/* reset response producers */
	for(i = 0; i < cell->num_response_producers; i++) {
		free_antigen(cell->response_producer[i]->antigen);
		cell->response_producer[i]->antigen = NULL;
	}
}

/* set the tissue from output of a cell */
void set_tissue(Tissue *tissue, const unsigned int index)
{
	Cell *cell  = tissue->cell[index];
	unsigned int i;

	/* adjust cytokine levels from cell cytokine producers */
	for(i = 0; i < cell->num_cytokine_producers; i++)
		if(cell->cytokine_producer[i]->type < tissue->num_cytokines)
			tissue->cytokine[cell->cytokine_producer[i]->type]->val += \
					cell->cytokine_producer[i]->val;
}

/* allow tissue and cells to interact */
void step_tissue(Tissue *tissue)
{
	unsigned int i;

	pthread_mutex_lock(tissue->lock);

	/* set input to cell */
	randomise_list(tissue->num_cells);
	for(i = 0; i < tissue->num_cells; i++)
		if(tissue->cell[randlist[i]]) {
			tissue->cell[randlist[i]]->iterations++;
			set_cell_input(tissue, randlist[i]);
			set_antigen_producers(tissue->cell[randlist[i]]);
		}

	/* cycle cells */
	randomise_list(tissue->num_cells);
	for(i = 0; i < tissue->num_cells; i++)
		if(tissue->cell[randlist[i]])
			tissue->cell[randlist[i]]->cycle(tissue->cell[randlist[i]]);

	/* set tissue from cell output */
	randomise_list(tissue->num_cells);
	for(i = 0; i < tissue->num_cells; i++)
		if(tissue->cell[randlist[i]]) {
			set_tissue(tissue, randlist[i]);
			produce_response(tissue->cell[randlist[i]]);
		}

	/* reduce cytokine levels in tissue */
	for(i = 0; i < tissue->num_cytokines; i++)
		tissue->cytokine[i]->val *= tissue->cytokine_decay_rate;

	pthread_mutex_unlock(tissue->lock);
}

/* try and set a value from the parameter file */
int read_parameter_uint(char *filename, char *parameter, unsigned int *val)
{
	FILE *fp;
	char line[1024], *ptr, *endptr[1];

	if(!(fp = fopen(filename, "r")))
		err(1, "error: couldn't read parameter file");

	while(fgets(line, sizeof(line), fp)) {
		if(line[0] == '#')  /* comment line */
			continue;
		if(!(ptr = index(line, '=')))
			err(1, "error: couldn't parse parameter file");
		*ptr = '\0';
		if(strcmp(line, parameter) == 0) {
			/* it's our parameter */
			*val = (unsigned int) strtol(ptr + 1, endptr, 10);
			if(**endptr != '\n')
				err(1, "edddrror: couldn't parse parameter file");
			fclose(fp);
			return 0;  /* found */
		}
	}

	fclose(fp);
	return 1;  /* not found */
}

/* save parameters in parameter file to log */
void log_parameter_file(char *filename)
{
	FILE *fp;
	char line[128], message[1024], *ptr;

	if(!(fp = fopen(filename, "r")))
		snprintf(message, sizeof(message) - 1, "using default parameters");
	else {
		ptr = message;
		ptr += snprintf(message, sizeof(message) - 1, \
				"using parameter file %s: ", filename);
		while(fgets(line, sizeof(line), fp)) {
			if(line[0] == '#')  /* comment line */
				continue;
			line[strlen(line) - 1] = '\0';
			ptr += sprintf(ptr, "%s ; ", line);  /* fix me */
		}
		*(ptr - 2) = '\0';
		fclose(fp);
	}

	/* save parameters to log */
	logtime_tissue(message);
}

/* initialise a probe - get data from tissue periodically */
void init_probe(Tissue *tissue, void (*probe)(Tissue *), unsigned long rate)
{
	if(!rate)  /* disable if rate=0 */
		return;

	tissue->probe = probe;
	tissue->probe_rate = rate;
	
	if(pthread_create(&tissue->probe_thread, NULL, handle_probe, \
			(void *) tissue))
		logfatal_tissue("couldn't create probe thread");	
}

/* thread callback for probe */
void *handle_probe(void *arg)
{
	Tissue *tissue = (Tissue *) arg;

	while(1) {
		/* do user supplied probe function */
		pthread_mutex_lock(tissue->lock);
		tissue->probe(tissue);
		pthread_mutex_unlock(tissue->lock);
		/* sleep a bit */
		usleep(tissue->probe_rate);
	}
}
